package com.xebialabs.xlrelease.domain;

import java.io.IOException;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.joda.time.DateTimeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.blackout.BlackoutMetadata;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.recover.TaskRecoverOp;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.PasswordStringVariable;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.domain.variables.reference.*;
import com.xebialabs.xlrelease.events.*;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.service.ExecuteAbortScriptAction;
import com.xebialabs.xlrelease.service.ExecuteFacetAction;
import com.xebialabs.xlrelease.service.ExecuteFailureHandlerAction;
import com.xebialabs.xlrelease.service.ExecutePreconditionAction;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xlrelease.utils.DateVariableUtils;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.XLRelease.PREFIX;
import static com.xebialabs.xlrelease.domain.ContainerTaskDefinition.isContainerTaskDefinition;
import static com.xebialabs.xlrelease.domain.FailureReasons.UNRESOLVED_VARIABLES;
import static com.xebialabs.xlrelease.domain.PythonScriptDefinition.isScriptDefinition;
import static com.xebialabs.xlrelease.domain.recover.TaskRecoverOp.RUN_SCRIPT;
import static com.xebialabs.xlrelease.domain.status.FlagStatus.OK;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.ABORTED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.ABORT_SCRIPT_IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED_IN_ADVANCE;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.FACET_CHECK_IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.FAILED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.FAILING;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.FAILURE_HANDLER_IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.PENDING;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.PLANNED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.PRECONDITION_IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.SKIPPED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.SKIPPED_IN_ADVANCE;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.WAITING_FOR_INPUT;
import static com.xebialabs.xlrelease.variable.VariableHelper.*;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.UUID.randomUUID;
import static java.util.stream.Collectors.*;
import static org.springframework.util.StringUtils.isEmpty;

/**
 * An Digital.ai Release task. See also {@link GateTask}.
 */
@Metadata(label = "Manual", versioned = false)
@PublicApiRef
@ShowOnlyPublicApiMembers
public class Task extends PlanItem implements Lockable {

    public static final String CATEGORY_INPUT = "input";
    public static final String CATEGORY_OUTPUT = "output";

    public static final double DUE_SOON_THRESHOLD = 0.75;

    private Integer releaseUid;

    private Integer ciUid;

    @Property(asContainment = true, required = false, description = "The comments on the task.")
    private List<Comment> comments = new ArrayList<>();

    @Property(asContainment = true, description = "The phase or task this task is contained in.")
    private TaskContainer container;

    @Property(asContainment = true, required = false, description = "Facets applied to the task.")
    private List<Facet> facets = new ArrayList<>();

    @Property(required = false, description = "List of file attachments on this task.")
    private List<Attachment> attachments = new ArrayList<>();

    @Property(description = "The state the task is in.")
    protected TaskStatus status;

    @Property(required = false, description = "The name of the team this task is assigned to.")
    protected String team;

    @Property(required = false, description = "The watchers assigned to this task.")
    private Set<String> watchers = new HashSet<>();

    @Property(required = false, defaultValue = "true", description = "The task is not started until the scheduledStartDate is reached if set to true.")
    protected boolean waitForScheduledStartDate;

    @Property(required = false, defaultValue = "false", description = "The task is to be delayed when a blackout period is active.")
    protected boolean delayDuringBlackout;

    @Property(required = false, category = "internal", defaultValue = "false", description = "The task is postponed by a blackout period")
    protected boolean postponedDueToBlackout;

    @Property(required = false, category = "internal", defaultValue = "false", description = "The task is waiting for environment(s) to be reserved")
    protected boolean postponedUntilEnvironmentsAreReserved;

    @Property(required = false, category = "internal", description = "The original scheduled start date.")
    protected Date originalScheduledStartDate;

    @Property(required = false, category = "internal")
    protected boolean hasBeenFlagged = false;

    @Property(required = false, category = "internal")
    protected boolean hasBeenDelayed = false;

    @Property(required = false, description = "A snippet of code that is evaluated when the task is started.")
    protected String precondition;

    @Property(required = false, description = "A snippet of code that is evaluated when the task is failed.")
    protected String failureHandler;

    // used to enable or disable from ui
    @Property(required = false, defaultValue = "false", description = "The failed script will be executed.")
    protected boolean taskFailureHandlerEnabled = false;

    @Property(required = false, description = "Task recovery operation performed after task failure.")
    protected TaskRecoverOp taskRecoverOp;

    @Property(description = "The number of times this task has failed.")
    protected int failuresCount = 0;

    @Property(required = false, category = "internal")
    protected String executionId;

    @Property(asContainment = true, required = false, description = "Map from property name to a variable name that replaces that property")
    protected Map<String, String> variableMapping = new HashMap<>();

    @Property(asContainment = true, required = false, description = "Similar to variableMapping, but only for password variables with external values and it is managed internally.")
    private Map<String, String> externalVariableMapping = new HashMap<>();

    @Property(hidden = true, category = "internal", description = "Maximum size of a comment on a task. Default value is 32768.", defaultValue = "32768")
    protected int maxCommentSize;

    @Property(required = false, description = "The tags of the task. Tags can be used for grouping and querying.")
    protected List<String> tags = new ArrayList<>();

    @Property(required = false, hidden = true, description = "URI of the HTML file to render the task")
    private String configurationUri;

    @Property(required = false, category = "internal")
    private boolean dueSoonNotified;

    @Property(required = false, defaultValue = "false", description = "The task is locked")
    private boolean locked = false;

    @Property(required = false, defaultValue = "false", description = "Check attributes on task execution")
    private boolean checkAttributes = false;

    private static final Logger logger = LoggerFactory.getLogger(Task.class);

    public String getConfigurationUri() {
        return configurationUri;
    }

    public void setConfigurationUri(final String configurationUri) {
        this.configurationUri = configurationUri;
    }

    public Integer getCiUid() {
        return ciUid;
    }

    public void setCiUid(final Integer ciUid) {
        this.ciUid = ciUid;
    }

    @PublicApiMember
    public List<Comment> getComments() {
        return comments;
    }

    @PublicApiMember
    public TaskContainer getContainer() {
        return container;
    }

    public void setContainer(TaskContainer container) {
        this.container = container;
    }

    @PublicApiMember
    public TaskStatus getStatus() {
        return status;
    }

    public void setStatus(TaskStatus status) {
        this.status = status;
    }

    @PublicApiMember
    public String getTeam() {
        return team;
    }

    public boolean hasTeam() {
        return this.team != null;
    }

    @PublicApiMember
    public void setTeam(String team) {
        this.team = team;
    }

    @PublicApiMember
    public Set<String> getWatchers() {
        return watchers;
    }

    @PublicApiMember
    public void setWatchers(Set<String> watchers) {
        this.watchers = watchers;
    }

    public void addWatcher(String watcher) {
        Set<String> updated = new HashSet<>(this.getWatchers());
        updated.add(watcher);
        setWatchers(updated);
    }

    public void removeWatcher(String watcher) {
        Set<String> updated = new HashSet<>(this.getWatchers());
        updated.remove(watcher);
        setWatchers(updated);
    }

    @PublicApiMember
    public void setPrecondition(String precondition) {
        this.precondition = precondition;
    }

    @PublicApiMember
    public String getPrecondition() {
        return precondition;
    }

    @PublicApiMember
    public String getFailureHandler() {
        return failureHandler;
    }

    @PublicApiMember
    public String getAbortScript() throws IOException {
        return null;
    }

    @PublicApiMember
    public void setFailureHandler(final String failureHandler) {
        this.failureHandler = failureHandler;
    }

    @PublicApiMember
    public boolean isTaskFailureHandlerEnabled() {
        return taskFailureHandlerEnabled;
    }

    @PublicApiMember
    public void setTaskFailureHandlerEnabled(final boolean taskFailureHandlerEnabled) {
        this.taskFailureHandlerEnabled = taskFailureHandlerEnabled;
    }


    @PublicApiMember
    public TaskRecoverOp getTaskRecoverOp() {
        return taskRecoverOp;
    }

    @PublicApiMember
    public void setTaskRecoverOp(final TaskRecoverOp taskRecoverOp) {
        this.taskRecoverOp = taskRecoverOp;
    }

    @PublicApiMember
    public List<String> getTags() {
        return tags;
    }

    @PublicApiMember
    public void setTags(List<String> tags) {
        this.tags = tags != null ? tags.stream().distinct().collect(Collectors.toList()) : Collections.emptyList();
    }

    @PublicApiMember
    public boolean isCheckAttributes() {
        return checkAttributes;
    }

    @PublicApiMember
    public void setCheckAttributes(boolean checkAttributes) {
        this.checkAttributes = checkAttributes;
    }

    public void setVariableMapping(final Map<String, String> variableMapping) {
        // create modifiable variable mapping
        this.variableMapping = new HashMap<>(variableMapping);
    }

    public Map<String, String> getVariableMapping() {
        return variableMapping;
    }

    public boolean hasVariableMapping() {
        return variableMapping != null && !variableMapping.isEmpty();
    }

    public void setMaxCommentSize(final int maxCommentSize) {
        this.maxCommentSize = maxCommentSize;
    }

    public int getMaxCommentSize() {
        return maxCommentSize;
    }

    public static boolean isDefaultTaskType(Type type) {
        return PREFIX.equals(type.getPrefix());
    }

    public static <T extends Task> T fromType(String taskType) {
        return fromType(Type.valueOf(taskType));
    }

    @SuppressWarnings("unchecked")
    public static <T extends Task> T fromType(Type taskType) {
        T task;
        if (isDefaultTaskType(taskType) || isContainerTaskDefinition(taskType)) {
            task = taskType.getDescriptor().newInstance(null);
        } else if (isScriptDefinition(taskType)) {
            PythonScript pythonScript = taskType.getDescriptor().newInstance(null);
            CustomScriptTask scriptTask = Type.valueOf(CustomScriptTask.class).getDescriptor().newInstance(null);
            scriptTask.setPythonScript(pythonScript);
            pythonScript.setCustomScriptTask(scriptTask);
            task = (T) scriptTask;
        } else {
            throw new IllegalArgumentException(String.format(
                    "Can only create tasks from default task types, container task types or subtypes of %s",
                    Type.valueOf(PythonScript.class)));
        }
        task.applyDefaults();
        return task;
    }

    protected void applyDefaults() {
    }

    public Phase getPhase() {
        if (container instanceof Task) {
            return ((Task) container).getPhase();
        }
        if (container instanceof Phase) {
            return (Phase) container;
        }
        if (container == null) {
            return null;
        }

        throw new IllegalStateException("Unexpected type for container: " + container.getClass().getName());
    }

    public Changes start() {
        if (isCompletedInAdvance()) {
            return markAsDone(getId(), COMPLETED);
        } else if (isSkippedInAdvance()) {
            return markAsDone(getId(), SKIPPED);
        } else if (canStartNow()) {
            return startNow(getId(), false);
        } else {
            return delayStartup();
        }
    }

    public boolean canStartNow() {
        return getScheduledStartDate() == null || (!isWaitForScheduledStartDate() && !isDelayDuringBlackout()) || new Date().after(getScheduledStartDate());
    }

    public boolean canScheduleTaskStart() {
        return !getRelease().isTemplate() && status.isOneOf(PLANNED, PENDING) && getScheduledStartDate() != null && (isWaitForScheduledStartDate() || isDelayDuringBlackout());
    }

    public Changes startPending(String targetId) {
        return startNow(targetId, true);
    }

    public Changes startWithInput() {
        checkState(isWaitingForInput(), "Task '%s' can only be started manually when it is waiting for input. It is now %s.", getTitle(), getStatus());
        return execute(getId(), new TaskStartOperation(this));
    }

    public Changes retry(String targetId) {
        checkState(!getId().equals(targetId) || isFailed() || isFailureHandlerInProgress() || isAbortScriptInProgress(),
                "Task '%s' can only be retried when it is failed, handling failure or executing abort script. It is now %s.", getTitle(), getStatus());
        postponedDueToBlackout = false;
        postponedUntilEnvironmentsAreReserved = false;
        return execute(targetId, new TaskRetryOperation(this));
    }

    @Override
    public void setScheduledStartDate(Date scheduledStartDate) {
        if ((this.getScheduledStartDate() != null && !this.getScheduledStartDate().equals(scheduledStartDate))
                || (this.getScheduledStartDate() == null && scheduledStartDate != null)) {
            this.postponedDueToBlackout = false;
            this.postponedUntilEnvironmentsAreReserved = false;
            this.originalScheduledStartDate = null;
        }
        super.setScheduledStartDate(scheduledStartDate);
    }

    protected Changes startNow(String targetId, boolean shouldBePending) {
        checkState(!shouldBePending || isPending(), "Task '%s' can only be started manually when it is pending. It is now %s.", getTitle(), getStatus());
        return execute(targetId, new TaskStartOperation(this));
    }

    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        if (mustDelayDueToBlackout()) {
            return delayUntilBlackoutEnd();
        } else {
            List<String> unboundVariables = getUnboundRequiredVariables();
            if (!unboundVariables.isEmpty()) {
                return askForInput(unboundVariables);
            } else if (shouldFacetBeChecked()) {
                return executeFacetCheck();
            } else if (shouldPreconditionBeChecked()) {
                return executePrecondition();
            } else {
                return executeTask(targetId, operation);
            }
        }
    }

    public boolean shouldFacetBeChecked() {
        return getFacets().stream().anyMatch(f -> f.hasProperty(ScriptHelper.SCRIPT_LOCATION_PROPERTY)) &&
                isCheckAttributes() && !this.status.isOneOf(PRECONDITION_IN_PROGRESS, FACET_CHECK_IN_PROGRESS);
    }

    private Changes executeFacetCheck() {
        Changes changes = new Changes();
        backupTaskIfNecessary(changes);
        this.status = FACET_CHECK_IN_PROGRESS;
        this.generateExecutionId();
        changes.update(this);
        freezeVariablesOrFailTask(changes);
        changes.addPostAction(new ExecuteFacetAction(this));

        return changes;
    }

    private Changes askForInput(final List<String> unboundVariables) {
        Changes changes = new Changes();

        backupTaskIfNecessary(changes);

        this.status = WAITING_FOR_INPUT;

        changes.update(this);
        changes.addOperation(new TaskWaitingForInputOperation(this, unboundVariables));

        return changes;
    }

    private Changes executePrecondition() {
        Changes changes = new Changes();

        backupTaskIfNecessary(changes);

        this.status = PRECONDITION_IN_PROGRESS;
        this.generateExecutionId();
        changes.update(this);
        freezeVariablesOrFailTask(changes);
        changes.addPostAction(new ExecutePreconditionAction(this));

        return changes;
    }

    private Changes executeTask(String targetId, TaskStartOrRetryOperation operation) {
        Changes changes = new Changes();

        checkOwnId(targetId);

        backupTaskIfNecessary(changes);

        this.status = IN_PROGRESS;
        if (!hasStartDate()) {
            this.setStartDate(new Date());
        }
        changes.addOperation(operation);
        freezeVariablesOrFailTask(changes);

        return changes;
    }

    private void freezeVariablesOrFailTask(Changes changes) {
        try {
            Set<String> unresolvedVariables = freezeVariables(changes, false);
            if (!unresolvedVariables.isEmpty()) {
                final Set<String> markdownVariables = unresolvedVariables.stream().map(variableName -> "`" + variableName + "`").collect(toSet());
                String variableNames = String.join(", ", markdownVariables);
                changes.addAll(fail(getId(), UNRESOLVED_VARIABLES.format(variableNames)));
            }
        } catch (Exception e) {
            changes.addAll(fail(getId(), e.getMessage()));
        }
    }

    /**
     * A task must be backed up only if:
     * - the task is a top level task (i.e not belonging to a task group)
     * - the task was not previously started
     */
    private void backupTaskIfNecessary(Changes changes) {
        if (canBeBackup()) {
            changes.addTaskToBackup(this);
        }
    }

    public boolean canBeBackup() {
        return (getPhase().equals(getContainer()) && this.status == PLANNED);
    }

    public boolean shouldPreconditionBeChecked() {
        return isPreconditionEnabled() && hasPrecondition() && !isPreconditionInProgress();
    }

    private boolean mustDelayDueToBlackout() {
        BlackoutMetadata metadata = getBlackoutMetadata();
        return isDelayDuringBlackout() && !isPreconditionInProgress() && metadata != null && metadata.isInBlackout(new Date());
    }

    private Changes delayUntilBlackoutEnd() {
        Changes changes = new Changes();

        if (!isPostponedUntilEnvironmentsAreReserved()) {
            // if it is already postponed due to facets - then we don't care
            // we'll wait for environments to align and check blackouts there again
            if (!isPostponedDueToBlackout() && getScheduledStartDate() != null) {
                setOriginalScheduledStartDate(scheduledStartDate);
            }
            setPostponedDueToBlackout(true);
            this.scheduledStartDate = getBlackoutMetadata().getEndOfBlackout(new Date());
            this.waitForScheduledStartDate = true;
            changes.addAll(delayStartup());
        }

        return changes;
    }

    public Changes postponeUntilEnvironmentsAreReserved(Date postponeUntil) {
        Changes changes = new Changes();

        if (!isPostponedDueToBlackout()) {
            // if it is already postponed due to blackout - then we don't care,
            // we'll wait for blackout to end first and then task will be re-executed
            if (!isPostponedUntilEnvironmentsAreReserved() && getScheduledStartDate() != null) {
                setOriginalScheduledStartDate(scheduledStartDate);
            }
            setPostponedUntilEnvironmentsAreReserved(true);
            this.scheduledStartDate = new Date(DateVariableUtils.truncateMilliseconds(postponeUntil.getTime()));
            this.waitForScheduledStartDate = true;
            this.startDate = null;
            this.setExecutionId(null);
            changes.addAll(delayStartup());
        }

        return changes;
    }

    private Changes delayStartup() {
        Changes changes = new Changes();

        backupTaskIfNecessary(changes);

        this.status = PENDING;
        changes.update(this);
        changes.addOperation(new TaskDelayOperation(this));

        return changes;
    }

    private BlackoutMetadata getBlackoutMetadata() {
        return (BlackoutMetadata) this.getRelease().get$metadata().get(BlackoutMetadata.BLACKOUT());
    }

    /**
     * @param targetId since tasks nest, this can either be the id of this task or the id of one of its children
     */
    public Changes markAsDone(String targetId, TaskStatus status) {
        checkOwnId(targetId);
        checkArgument(status.isOneOf(COMPLETED, SKIPPED, COMPLETED_IN_ADVANCE, SKIPPED_IN_ADVANCE), "Status is %s but must be either COMPLETED, SKIPPED, COMPLETE_IN_ADVANCE or SKIPPED_IN_ADVANCE", status);

        Changes changes = new Changes();
        boolean endRecovery = isFailureHandlerInProgress();
        boolean abortScriptInProgress = isAbortScriptInProgress();

        logger.debug("markAsDone(" + this.title + "): " + this.status + " -> " + status);

        TaskStatus previousStatus = getStatus();
        setStatus(status);
        if (executionId != null) {
            changes.addOperation(new CompleteTaskJobExecutionOperation(this, executionId));
            setExecutionId(null);
        }
        setStartAndEndDatesIfEmpty();
        if (isOverdue()) {
            hasBeenDelayed = true;
        }

        if (status == COMPLETED) {
            freezeVariables(changes, true);
            resetFlag();
            changes.update(this);
            if (previousStatus != COMPLETED_IN_ADVANCE) {
                changes.addOperation(new TaskCompleteOperation(this, false));
            }
        } else if (status == SKIPPED) {
            freezeVariables(changes, true);
            resetFlag();
            changes.update(this);
            if (previousStatus != SKIPPED_IN_ADVANCE) {
                changes.addOperation(new TaskSkipOperation(this, false));
            }
            if (endRecovery) {
                changes.addOperation(new TaskEndRecoveryOperation(this));
            } else if (abortScriptInProgress) {
                changes.addOperation(new TaskCompleteAbortScriptOperation(this));
            }
        } else if (status == COMPLETED_IN_ADVANCE) {
            setStartDate(getEndDate());
            changes.update(this);
            changes.addOperation(new TaskCompleteOperation(this, true));
        } else if (status == SKIPPED_IN_ADVANCE) {
            setStartDate(getEndDate());
            changes.update(this);
            changes.addOperation(new TaskSkipOperation(this, true));
        }

        cleanupExternalVariableUse();
        return changes;
    }

    public Changes fail(String targetId, String failReason) {
        return fail(targetId, failReason, User.AUTHENTICATED_USER, false);
    }

    public Changes fail(String targetId, String failReason, User user) {
        return fail(targetId, failReason, user, false);
    }

    public Changes fail(String targetId, String failReason, boolean fromAbort) {
        return fail(targetId, failReason, User.AUTHENTICATED_USER, fromAbort);
    }


    public Changes fail(String targetId, String failReason, User user, boolean fromAbort) {
        checkOwnId(targetId);
        cleanupExternalVariableUse();
        Changes changes = new Changes();
        //do not fail again if is already failed
        if (isFailing() || isFailed()) {
            return changes;
        }
        changes.update(this);
        if (executionId != null) {
            changes.addOperation(new CompleteTaskJobExecutionOperation(this, executionId));
            setExecutionId(null);
        }
        changes.addComment(this, user, failReason);
        changes.addOperation(new TaskFailOperation(this, failReason));
        if (fromAbort && hasAbortScript() && !isAbortScriptInProgress() && !isDefunct()) {
            failuresCount++;
            setStatus(ABORT_SCRIPT_IN_PROGRESS);
            generateExecutionId();
            changes.addOperation(new TaskStartAbortScriptOperation(this));
            changes.addPostAction(new ExecuteAbortScriptAction(this));
        } else if (isTaskFailureHandlerEnabled() && isFailureHandlerEnabled() && !isFailureHandlerInProgress() && !isDefunct()) {
            if (hasFailureHandlerScript() && RUN_SCRIPT == taskRecoverOp && getRelease().getScriptUsername() == null) {
                changes.addComment(this, User.SYSTEM, "Failure handler script could not run because no scriptUser is defined on Release level.");
                setStatus(FAILED);
                return changes;
            } else {
                failuresCount++;
                setStatus(FAILURE_HANDLER_IN_PROGRESS);
                generateExecutionId();
                changes.addOperation(new TaskStartRecoveryOperation(this));
                changes.addPostAction(new ExecuteFailureHandlerAction(this));
            }
        } else if (!isDefunct()) {
            boolean endRecovery = isFailureHandlerInProgress();
            boolean abortScriptInProgress = isAbortScriptInProgress();
            failuresCount++;
            setStatus(FAILED);
            if (endRecovery) {
                changes.addOperation(new TaskEndRecoveryOperation(this));
            } else if (abortScriptInProgress) {
                changes.addOperation(new TaskCompleteAbortScriptOperation(this));
            }
        }

        return changes;
    }

    public Changes reopen() {
        setStatus(PLANNED);
        setEndDate(null);
        setStartDate(null);

        Changes changes = new Changes();
        changes.update(this);
        changes.addOperation(new TaskReopenOperation(this));

        return changes;
    }

    private void checkOwnId(String targetId) {
        checkArgument(getId().equals(targetId), "Attempt to access subtask '%s' of leaf task '%s'", targetId, getId());
    }

    public Changes abort() {
        Changes changes = new Changes();
        if (!isDone()) {
            setStatus(ABORTED);
            if (!hasStartDate()) {
                setStartDate(new Date());
            }
            setEndDate(new Date());
            changes.addOperation(new TaskAbortOperation(this));
            changes.update(this);
        }
        return changes;
    }

    public boolean hasBeenStarted() {
        return status != null && status.hasBeenStarted();
    }

    @PublicApiMember
    public boolean hasBeenFlagged() {
        return hasBeenFlagged;
    }

    public int getFlaggedCount() {
        return hasBeenFlagged() ? 1 : 0;
    }

    public int getDelayedCount() {
        return hasBeenDelayed() ? 1 : 0;
    }

    @PublicApiMember
    public boolean hasBeenDelayed() {
        return hasBeenDelayed;
    }

    @PublicApiMember
    public int getFailuresCount() {
        return failuresCount;
    }

    public void setHasBeenFlagged(boolean hasBeenFlagged) {
        this.hasBeenFlagged = hasBeenFlagged;
    }

    public void setHasBeenDelayed(boolean hasBeenDelayed) {
        this.hasBeenDelayed = hasBeenDelayed;
    }

    public void setFailuresCount(int failuresCount) {
        this.failuresCount = failuresCount;
    }

    public boolean isDone() {
        return status != null && status.isDone();
    }

    public boolean isDoneInAdvance() {
        return status != null && status.isDoneInAdvance();
    }

    public boolean isDefunct() {
        return isAborted() || isDone();
    }

    public boolean isUpdatable() {
        return !isDefunct() && !isDoneInAdvance();
    }

    public boolean isAborted() {
        return status == ABORTED;
    }

    public boolean isNotYetReached() {
        return isPlanned() || isDoneInAdvance();
    }

    public boolean isPlanned() {
        return status != null && status == PLANNED;
    }

    public boolean isActive() {
        return status != null && status.isActive();
    }

    public boolean isInProgress() {
        return status == IN_PROGRESS;
    }

    public boolean isPending() {
        return status == PENDING;
    }

    public boolean isWaitingForInput() {
        return status == WAITING_FOR_INPUT;
    }

    public boolean isFailed() {
        return status == FAILED;
    }

    public boolean isFailing() {
        return status == FAILING;
    }

    public boolean isCompletedInAdvance() {
        return status == COMPLETED_IN_ADVANCE;
    }

    public boolean isSkipped() {
        return status == SKIPPED;
    }

    public boolean isSkippedInAdvance() {
        return status == SKIPPED_IN_ADVANCE;
    }

    public boolean isPreconditionInProgress() {
        return status == PRECONDITION_IN_PROGRESS;
    }

    public boolean isFailureHandlerInProgress() {
        return status == FAILURE_HANDLER_IN_PROGRESS;
    }

    public boolean isAbortScriptInProgress() {
        return status == ABORT_SCRIPT_IN_PROGRESS;
    }

    public boolean isFacetInProgress() {
        return status == FACET_CHECK_IN_PROGRESS;
    }

    private boolean hasPrecondition() {
        return !isEmpty(precondition);
    }

    protected boolean hasFailureHandlerScript() {
        return taskFailureHandlerEnabled && !Strings.isBlank(failureHandler);
    }

    public boolean hasAbortScript() {
        return false;
    }

    protected boolean hasTaskRecoverOp() {
        return taskRecoverOp != null;
    }

    public boolean isMovable() {
        return isPlanned() || isDoneInAdvance();
    }

    public boolean isAssignedTo(Team team) {
        return team.getTeamName().equals(getTeam());
    }

    public boolean isGate() {
        return this instanceof GateTask;
    }

    public boolean isTaskGroup() {
        return this instanceof TaskGroup;
    }

    public boolean isParallelGroup() {
        return this instanceof ParallelGroup;
    }

    @PublicApiMember
    public List<Attachment> getAttachments() {
        return attachments;
    }

    public void setAttachments(List<Attachment> attachments) {
        this.attachments = attachments;
    }

    public boolean isPreconditionEnabled() {
        return getProperty("preconditionEnabled");
    }

    public boolean isFailureHandlerEnabled() {
        return getProperty("failureHandlerEnabled");
    }

    public String getExecutionId() {
        return executionId;
    }

    public void setExecutionId(String executionId) {
        this.executionId = executionId;
    }

    public void generateExecutionId() {
        setExecutionId(randomUUID().toString());
    }

    public void deleteAttachment(String attachmentId) {
        CiHelper.removeCisWithId(attachments, attachmentId);
    }

    public Changes resetToPlanned() {
        Changes changes = new Changes();

        setStatus(PLANNED);
        setStartDate(null);
        setEndDate(null);
        setOverdueNotified(false);
        setDueSoonNotified(false);
        setFailuresCount(0);
        setHasBeenFlagged(false);
        setHasBeenDelayed(false);

        changes.update(this);

        for (Comment comment : comments) {
            changes.remove(comment.getId());
        }
        comments.clear();

        return changes;
    }

    @Override
    public void setFlagStatus(FlagStatus flagStatus) {
        super.setFlagStatus(flagStatus);

        Release release = getRelease();
        if (release != null) {
            release.updateRealFlagStatus();
        }

        if (flagStatus != FlagStatus.OK) {
            this.hasBeenFlagged = true;
        }
    }

    public Release getRelease() {
        return getPhase() != null ? getPhase().getRelease() : null;
    }

    @Override
    public Integer getReleaseUid() {
        if (releaseUid != null) {
            return releaseUid;
        } else {
            Release release = getRelease();
            if (null != release) {
                return release.getCiUid();
            } else {
                return null;
            }
        }
    }

    @Override
    public void setReleaseUid(Integer releaseUid) {
        this.releaseUid = releaseUid;
    }

    public String getDisplayPath() {
        return getId() + " (" + getTitle() + ")";
    }

    public String getReleaseOwner() {
        return getPhase().getReleaseOwner();
    }

    public List<Task> getAllTasks() {
        return singletonList(this);
    }

    public List<PlanItem> getChildren() {
        return new ArrayList<>();
    }

    public void accept(ReleaseVisitor visitor) {
        visitor.visit(this);
        facets.forEach(facet -> facet.accept(visitor));
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        return asList(
                new PropertyUsagePoint(this, "title"),
                new PropertyUsagePoint(this, "description"),
                new PropertyUsagePoint(this, "owner"),
                new PropertyUsagePoint(this, "precondition")
        );
    }

    private void populateExternalVariableMapping() {
        if (variableMapping.isEmpty()) {
            return;
        }

        for (Map.Entry<String, String> entry : variableMapping.entrySet()) {
            String variableName = entry.getValue();
            Optional<Variable> maybeVariable = resolveVariable(variableName);
            if (maybeVariable.isPresent()) {
                Variable variable = maybeVariable.get();
                if (variable.isPassword() && ((PasswordStringVariable) variable).getExternalVariableValue() != null) {
                    externalVariableMapping.put(entry.getKey(), variableName);
                }
            }
        }
    }

    private void cleanupExternalVariableUse() {
        if (externalVariableMapping.isEmpty()) {
            return;
        }

        for (Map.Entry<String, String> entry : externalVariableMapping.entrySet()) {
            String fqPropertyName = entry.getKey();
            Optional<CiProperty> ciPropertyOptional = CiProperty.of(this, fqPropertyName);
            if (!ciPropertyOptional.isPresent()) {
                continue;
            }
            CiProperty ciProperty = ciPropertyOptional.get();
            ciProperty.setValue(null);

            String variableName = entry.getValue();
            Optional<Variable> maybeVariable = resolveVariable(variableName);
            if (maybeVariable.isPresent()) {
                Variable variable = maybeVariable.get();
                variable.setUntypedValue(null);
            }

            // No need to restore the mapping for completed tasks
            if (status != COMPLETED) {
                variableMapping.put(fqPropertyName, variableName);
            }
        }
        externalVariableMapping.clear();
    }

    public Set<String> freezeVariables(Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = new HashSet<>();
        Release release = getPhase().getRelease();
        populateExternalVariableMapping();
        resolveExternalVariables();

        Map<String, ValueWithInterpolation> variables = release.getAllVariableValuesAsStringsWithInterpolationInfo();
        Map<String, String> passwordVariables = release.getPasswordVariableValues();

        changes.update(this);

        setTitle(replaceVariablesInTitle(release, variables, unresolvedVariables, freezeEvenIfUnresolved));
        setDescription(replaceAllWithInterpolation(getDescription(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setOwner(replaceAllWithInterpolation(getOwner(), variables, unresolvedVariables, freezeEvenIfUnresolved));

        if (this.watchers != null) {
            setWatchers(getWatchers().stream().map(w -> replaceAllWithInterpolation(w, variables, unresolvedVariables, freezeEvenIfUnresolved)).collect(toSet()));
        }

        if (isPreconditionEnabled()) {
            setPrecondition(replaceAllWithInterpolation(precondition, variables, unresolvedVariables, freezeEvenIfUnresolved));
        }

        if (hasFailureHandlerScript() && RUN_SCRIPT == taskRecoverOp) {
            setFailureHandler(replaceAllWithInterpolation(failureHandler, variables, unresolvedVariables, freezeEvenIfUnresolved));
        }

        unresolvedVariables.addAll(freezeInputVariableMapping(changes));
        unresolvedVariables.addAll(freezeVariablesInCustomFields(variables, passwordVariables, changes, freezeEvenIfUnresolved));
        facets.forEach(facet -> {
            unresolvedVariables.addAll(facet.freezeVariables(variables, changes, freezeEvenIfUnresolved));
            unresolvedVariables.addAll(freezeInputVariableMapping(changes, facet.getVariableMapping(), facet));
        });

        if (release.isAllowPasswordsInAllFields()) {
            //passwords will be substituted at the very end when the task is executed.
            //remove unresolved password variables since they will be added later.
            unresolvedVariables.removeAll(passwordVariables.keySet());
        }

        return unresolvedVariables;
    }

    private String replaceVariablesInTitle(final Release release, final Map<String, ValueWithInterpolation> stringVariables, final Set<String> unresolvedVariables, final boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedTitleVariables = new HashSet<>();
        String title = replaceAllWithInterpolation(getTitle(), filterOutBlankValues(stringVariables), unresolvedTitleVariables, freezeEvenIfUnresolved);
        List<Variable> allVariables = release.getAllVariables();
        for (String titleVarKey : unresolvedTitleVariables) {
            allVariables.stream()
                    .filter(var -> var.getKey().equals(withoutVariableSyntax(titleVarKey)))
                    .filter(Variable::getRequiresValue)
                    .findFirst()
                    .ifPresent(unresolvedRequiredVariable -> unresolvedVariables.add(titleVarKey));
        }
        return title;
    }

    private Set<String> freezeInputVariableMapping(Changes changes) {
        return freezeInputVariableMapping(changes, variableMapping, this);
    }

    private Set<String> freezeInputVariableMapping(Changes changes, Map<String, String> variableMapping, ConfigurationItem rootCi) {
        Set<String> unresolvedVariables = new HashSet<>();

        Iterator<String> iterator = variableMapping.keySet().iterator();
        while (iterator.hasNext()) {
            String fqPropertyName = iterator.next();
            Optional<CiProperty> ciPropertyOptional = CiProperty.of(rootCi, fqPropertyName);
            if (!ciPropertyOptional.isPresent()) {
                continue;
            }
            CiProperty ciProperty = ciPropertyOptional.get();

            if (shouldFreezeVariableMapping(ciProperty)) {
                Optional<Variable> maybeVariable = resolveVariable(variableMapping.get(fqPropertyName));
                if (maybeVariable.isPresent()) {
                    Variable variable = maybeVariable.get();
                    if (!variable.isValueEmpty() || !variable.getRequiresValue()) {
                        ciProperty.setValue(variable.getInternalValue());
                        changes.update(ciProperty.getParentCi());
                    } else {
                        unresolvedVariables.add(variableMapping.get(fqPropertyName));
                    }
                } else {
                    unresolvedVariables.add(variableMapping.get(fqPropertyName));
                }
                iterator.remove();
            }
        }
        return unresolvedVariables;
    }

    protected boolean shouldFreezeVariableMapping(CiProperty property) {
        return true;
    }

    private Optional<Variable> resolveVariable(final String variable) {
        Release release = getPhase().getRelease();
        String variableKey = VariableHelper.withoutVariableSyntax(variable);

        if (variableKey == null) {
            logger.debug("variableKey is null for variable {}", variable);
        }
        if (release.getGlobalVariables() == null) {
            logger.debug("Release global variables is null");
        }
        if (release.getFolderVariables() == null) {
            logger.debug("Release folder variables is null");
        }

        return Stream.<java.util.function.Supplier<Optional<Variable>>>of(
                () -> Optional.ofNullable(release.getVariablesByKeys().get(variableKey)),
                () -> Optional.ofNullable(release.getGlobalVariables()).map(gv -> gv.getVariablesByKeys().get(variableKey)),
                () -> Optional.ofNullable(release.getFolderVariables()).map(fv -> fv.getVariablesByKeys().get(variableKey))
        ).map(java.util.function.Supplier::get).filter(Optional::isPresent).map(Optional::get).findFirst();
    }

    public List<Variable> getInputVariables() {
        return getReferencedVariables(input -> {
                    boolean isNotOutputVariable = input.getType() != VariableReference.VariableUsageType.SCRIPT_RESULT;
                    boolean isUsedOutsideUserInputTask = input.getUsagePoints().stream().anyMatch(usagePoint -> !(usagePoint instanceof UserInputTaskUsagePoint));
                    return isUsedOutsideUserInputTask && isNotOutputVariable;
                }
        );
    }

    public List<Variable> getReferencedVariables() {
        return getReferencedVariables((x) -> true);
    }

    private List<Variable> getReferencedVariables(Predicate<VariableReference> filter) {
        final Set<String> keys = getReferencedVariableKeys(filter);
        List<Variable> releaseVariables = getRelease().getVariables();
        return releaseVariables.stream().filter(input -> keys.contains(input.getKey())).collect(toList());
    }

    private Set<String> getReferencedVariableKeys(Predicate<VariableReference> filter) {
        Set<String> variableKeys = collectVariableReferences().stream()
                .filter(filter).map(input -> VariableHelper.withoutVariableSyntax(input.getKey())).collect(toSet());
        Set<String> variablesKeysInNonInterpolatableValues = getRelease().getVariablesKeysInNonInterpolatableVariableValues();
        variableKeys.removeAll(variablesKeysInNonInterpolatableValues);
        return variableKeys;
    }

    protected List<String> getUnboundRequiredVariables() {
        final List<String> unbound = new ArrayList<>();
        getInputVariables().forEach(v -> {
            if (v.getRequiresValue() && v.isValueEmpty()) unbound.add(withVariableSyntax(v.getKey()));
        });
        return unbound;
    }

    private void resolveExternalVariables() {
        final Release release = getRelease();
        if (release == null) {
            return;
        }

        //all passwords in scope
        final Map<String, PasswordStringVariable> externalVariables = new HashMap<>(VariableHelper.getUsedExternalPasswordVariables(release));

        //only keep the passwords actually used
        var usedVariables = getReferencedVariableKeys(it -> true);
        externalVariables.keySet().retainAll(usedVariables);

        //group variables by their server so they can be bulk resolved
        final Map<ExternalVariableServer, List<PasswordStringVariable>> variablesByServer = externalVariables.values()
                .stream()
                .collect(Collectors.groupingBy(v ->
                        checkNotNull(v.getExternalVariableValue().getServer(),
                                "Unable to load configuration for variable \"%s\" with Id \"%s\"", v.getKey(), v.getId())));

        for (Map.Entry<ExternalVariableServer, List<PasswordStringVariable>> e : variablesByServer.entrySet()) {
            ExternalVariableServer server = e.getKey();
            try {
                Map<String, String> externalValues = server.lookup(e.getValue());

                for (Map.Entry<String, String> e2 : externalValues.entrySet()) {
                    String variableName = e2.getKey();
                    String variableValue = e2.getValue();

                    PasswordStringVariable variable = externalVariables.get(variableName);
                    String encryptedValue = PasswordEncrypter.getInstance().ensureEncrypted(variableValue);

                    //mutates the blank value with a real value at this point. Treat it as a regular password downstream.
                    variable.setValue(encryptedValue);
                }
            } catch (Exception ex) {
                // This same code path is called through many different task states but we only want to actually
                // throw when the task is in progress
                if (isInProgress()) {
                    throw ex;
                }
            }
        }
    }

    /**
     * <p>
     * To be overridden in child classes with custom fields that can contain variables.
     * </p>
     * Apply {@link VariableHelper#replaceAll(Object, Map, Set, boolean)} to each field; if child entities are modified, add them
     * to {@code changes}.
     */
    @SuppressWarnings("unused")
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables,
                                                     Map<String, String> passwordVariables,
                                                     Changes changes,
                                                     boolean freezeEvenIfUnresolved) {
        return emptySet();
    }

    public boolean isWaitForScheduledStartDate() {
        return waitForScheduledStartDate;
    }

    public void setWaitForScheduledStartDate(boolean waitForScheduledStartDate) {
        this.waitForScheduledStartDate = waitForScheduledStartDate;
    }

    public boolean isDelayDuringBlackout() {
        return delayDuringBlackout;
    }

    public void setDelayDuringBlackout(final boolean delayDuringBlackout) {
        if (!delayDuringBlackout) {
            this.postponedDueToBlackout = false;
        }
        this.delayDuringBlackout = delayDuringBlackout;
    }

    private void resetFlag() {
        setFlagStatus(OK);
        setFlagComment("");
    }

    public void checkDatesValidity() {
        checkDatesValidity(scheduledStartDate, dueDate, plannedDuration);
    }

    public static final Predicate<Task> IS_AUTOMATED_AND_IN_PROGRESS = task -> task.isAutomated() && task.isInProgress();

    public boolean isAutomated() {
        return getProperty("automated");
    }

    public boolean ownerHasBeenReassigned(Task task) {
        return !Objects.equals(this.getOwner(), task.getOwner());
    }

    public boolean teamHasBeenReassigned(Task task) {
        return !Objects.equals(this.getTeam(), task.getTeam());
    }

    public boolean delayDuringBlackoutHasChanged(Task task) {
        return this.isDelayDuringBlackout() != task.isDelayDuringBlackout();
    }

    public boolean failureHandlerHasChanged(Task task) {
        return this.getTaskRecoverOp() != task.getTaskRecoverOp() || !Objects.equals(this.getFailureHandler(), task.getFailureHandler());
    }

    public boolean preconditionHasChanged(Task task) {
        return !Objects.equals(this.getPrecondition(), task.getPrecondition());
    }

    public Type getTaskType() {
        return this instanceof CustomScriptTask
                ? ((CustomScriptTask) this).getPythonScript().getType()
                : this.type;
    }

    public boolean isStillExecutingScript(String executionId) {
        return (isInProgress() || isPreconditionInProgress() || isFailureHandlerInProgress() || isAbortScriptInProgress() || isFacetInProgress())
                && Objects.equals(this.executionId, executionId);
    }

    private Set<VariableReference> collectVariableReferences() {
        return VariableCollectingVisitor.collectFrom(this);
    }

    public boolean isPostponedDueToBlackout() {
        return postponedDueToBlackout;
    }

    public void setPostponedDueToBlackout(boolean delay) {
        this.postponedDueToBlackout = delay;
    }

    public Date getOriginalScheduledStartDate() {
        return originalScheduledStartDate;
    }

    public void setOriginalScheduledStartDate(Date originalDate) {
        this.originalScheduledStartDate = originalDate;
    }

    public boolean isDueSoon() {
        double elapsedDurationFraction = getElapsedDurationFraction();
        return !isOverdue() && elapsedDurationFraction >= DUE_SOON_THRESHOLD;
    }

    public double getElapsedDurationFraction() {
        Date startDate = getStartOrScheduledDate();
        if (startDate == null) return 0.0d;
        Date now = new Date(DateTimeUtils.currentTimeMillis());

        return getOrCalculateDueDate().map(dueDate -> getElapsedDurationFraction(startDate, now, dueDate)).orElse(0.0);
    }

    private Double getElapsedDurationFraction(final Date startDate, final Date now, final Date dueDate) {
        long startDateTime = startDate.getTime();
        long nowTime = now.getTime();
        long dueDateTime = dueDate.getTime();

        long millisecondsPassedSinceStartDate = nowTime - startDateTime;
        long startDateTillDueDateDuration = dueDateTime - startDateTime;

        return (double) millisecondsPassedSinceStartDate / startDateTillDueDateDuration;
    }

    public boolean shouldNotifyDueSoon() {
        return !dueSoonNotified && isDueSoon();
    }

    public boolean isDueSoonNotified() {
        return dueSoonNotified;
    }

    public void setDueSoonNotified(final boolean dueSoonNotified) {
        this.dueSoonNotified = dueSoonNotified;
    }

    public void deleteTask(Task task) {
        //nothing to do
    }

    public void replaceTask(Task task) {
        //nothing to do
    }

    public Comment findComment(String commentId) {
        return comments.stream().filter(comment -> comment.getId().equals(commentId)).findFirst().orElse(null);
    }

    public void updateComment(Comment originalComment, Comment updatedComment) {
        int index = comments.indexOf(originalComment);
        if (index != -1) {
            comments.set(index, updatedComment);
        }
    }

    public void clearComments() {
        comments.clear();
    }

    @Override
    public boolean isLocked() {
        return locked;
    }

    @Override
    public void setLocked(final boolean locked) {
        this.locked = locked;
    }

    @Override
    public void lock() {
        this.setLocked(true);
    }

    @Override
    public void unlock() {
        this.setLocked(false);
    }

    public List<Facet> getFacets() {
        return facets;
    }

    public void setFacets(final List<Facet> facets) {
        this.facets = facets;
    }

    @PublicApiMember
    public String getUrl() {
        return String.format(
                "%s?openTaskDetailsModal=%s",
                getRelease().getUrl(),
                ServerUrl.convertToViewId(getId())
        );
    }

    public boolean isPostponedUntilEnvironmentsAreReserved() {
        return postponedUntilEnvironmentsAreReserved;
    }

    public Task setPostponedUntilEnvironmentsAreReserved(boolean postponedUntilEnvironmentsAreReserved) {
        this.postponedUntilEnvironmentsAreReserved = postponedUntilEnvironmentsAreReserved;
        return this;
    }
}
