package com.xebialabs.xlrelease.domain.delivery;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Ids;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(versioned = false)
public class Stage extends BaseConfigurationItem {

    public static final String DEFAULT_STAGE_TITLE = "New Stage";

    @Property
    private String title;

    @Property(defaultValue = "OPEN")
    private StageStatus status = StageStatus.OPEN;

    @Property(asContainment = true)
    private List<StageTrackedItem> items = new ArrayList<>();

    @Property(required = false)
    private Transition transition;

    @Property(description="The owner of the stage.")
    private String owner;

    @Property(description = "The name of the team this stage is assigned to.")
    private String team;

    public Stage() {

    }

    public Stage(final String title) {
        this.title = title;
    }

    public Stage(final String id, final String title) {
        this.id = id;
        this.title = title;
    }

    public StageTrackedItem getItemById(String trackedItemId) {
        return findItemById(trackedItemId).orElseThrow(() ->
                new LogFriendlyNotFoundException("Tracked item '%s' does not exist in stage '%s'", trackedItemId, title)
        );
    }

    public Optional<StageTrackedItem> findItemById(String trackedItemId) {
        return items.stream()
                .filter(stageItem -> Ids.getName(stageItem.getTrackedItemId()).equals(Ids.getName(trackedItemId)))
                .findFirst();
    }

    public void removeTrackedItem(String itemId) {
        items.removeIf(item -> Ids.getName(item.getTrackedItemId()).equals(Ids.getName(itemId)));
    }

    public void addTrackedItems(List<StageTrackedItem> items) {
        this.items.addAll(items);
    }

    public boolean isOpen() {
        return status == StageStatus.OPEN;
    }

    public boolean isClosed() {
        return status == StageStatus.CLOSED;
    }

    // ---

    @PublicApiMember
    public String getTitle() {
        return title;
    }

    @PublicApiMember
    public void setTitle(final String title) {
        this.title = title;
    }

    @PublicApiMember
    public StageStatus getStatus() {
        return status;
    }

    @PublicApiMember
    public void setStatus(final StageStatus status) {
        this.status = status;
    }

    @PublicApiMember
    public List<StageTrackedItem> getItems() {
        return items;
    }

    public void setItems(final List<StageTrackedItem> items) {
        this.items = items;
    }

    public boolean hasTransition() {
        return getTransition() != null;
    }

    @PublicApiMember
    public String getOwner() {
        return owner;
    }

    @PublicApiMember
    public void setOwner(String value) {
        this.owner = value;
    }

    public boolean hasOwner() {
        return owner != null;
    }

    public boolean isOwner(String stageOwner) {
        return hasOwner() && getOwner().equalsIgnoreCase(stageOwner);
    }

    @PublicApiMember
    public String getTeam() {
        return team;
    }

    public boolean hasTeam() {
        return this.team != null;
    }

    @PublicApiMember
    public void setTeam(String team) {
        this.team = team;
    }

    @PublicApiMember
    public Transition getTransition() {
        return transition;
    }

    @PublicApiMember
    public void setTransition(Transition transition) {
        this.transition = transition;
        if (transition != null) {
            this.transition.setStage(this);
        }
    }
}
