package com.xebialabs.xlrelease.domain.variables.reference

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType
import com.xebialabs.xlrelease.domain.variables.reference.VariableUsagePoint.replace
import com.xebialabs.xlrelease.repository.CiProperty
import com.xebialabs.xlrelease.variable.VariableHelper

import java.util
import java.util.{Collections, Date}
import scala.jdk.CollectionConverters._

class VariableUsagePoint(variable: Variable) extends UsagePoint {

  def collectVariables(): util.Map[String, VariableReference.VariableUsageType] = {
    variable match {
      case v: StringVariable if v.isPreventInterpolation =>
        Collections.emptyMap()
      case _ =>
        VariableHelper.collectVariables(variable.getValue).asScala.map(_ -> VariableUsageType.DEFAULT).toMap.asJava
    }
  }

  def replaceVariable(toReplace: Variable, replacement: VariableOrValue): util.Set[ConfigurationItem] = {
    val replaceKey = VariableHelper.formatVariableIfNeeded(toReplace.getKey)
    Option(replacement.getVariable)
      .map(key => replaceString(replaceKey, VariableHelper.formatVariableIfNeeded(key)))
      .getOrElse(replaceValue(replaceKey, replacement.getValue))
      .toSet
      .map((v: Variable) => v: ConfigurationItem)
      .asJava
  }

  def getTargetProperty: CiProperty = variable match {
    case v: StringVariable => CiProperty.of(v, "value").orElse(null)
    case v: ListStringVariable => CiProperty.of(v, "value").orElse(null)
    case v: SetStringVariable => CiProperty.of(v, "value").orElse(null)
    case v: MapStringStringVariable => CiProperty.of(v, "value").orElse(null)
    case _ => null
  }

  protected def replaceString(key: String, value: String): Option[Variable] = {
    if (variable.isValueEmpty) {
      None
    } else {
      replace(key, value)
        .lift
        .apply(variable)
        .map(_ => variable)
    }
  }

  protected def replaceValue(key: String, value: AnyRef): Option[Variable] = value match {
    case v: String =>
      replaceString(key, v)
    case v: util.List[String@unchecked] =>
      val _v = new ListStringVariable()
      _v.setValue(v)
      replaceString(key, _v.getValueAsString)
    case v: util.Set[String@unchecked] =>
      val _v = new SetStringVariable()
      _v.setValue(v)
      replaceString(key, _v.getValueAsString)
    case v: util.Map[String@unchecked, String@unchecked] =>
      val _v = new MapStringStringVariable()
      _v.setValue(v)
      replaceString(key, _v.getValueAsString)
    case v: Date =>
      val _v = new DateVariable()
      _v.setValue(v)
      replaceString(key, _v.getValueAsString)
    case v: Integer =>
      val _v = new IntegerVariable()
      _v.setValue(v)
      replaceString(key, _v.getValueAsString)
    case _ =>
      // TODO: log.warn
      None
  }

}

object VariableUsagePoint {

  protected def replace(key: String, newKey: String): PartialFunction[Variable, Unit] = {
    case v: StringVariable =>
      v.setValue(v.getValue.replace(key, newKey))
    case v: ListStringVariable =>
      v.setValue(v.getValue.asScala.map(s => s.replace(key, newKey)).asJava)
    case v: SetStringVariable =>
      v.setValue(v.getValue.asScala.map(s => s.replace(key, newKey)).asJava)
    case v: MapStringStringVariable =>
      v.setValue(v.getValue.asScala.view.mapValues(s => s.replace(key, newKey)).toMap.asJava)
  }

}

