package com.xebialabs.xlrelease.domain;

import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import com.google.common.collect.ComparisonChain;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;

import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(description = "Log of something that happened in XL Release.", root = APPLICATIONS, versioned = false)
public class ActivityLogEntry extends BaseConfigurationItem {

    @Property
    private String username;

    @Property
    private String activityType;

    @Property
    private String message;

    @Property
    private Date eventTime;

    @Property
    private String targetType;

    @Property
    private String targetId;

    @Property(required = false)
    private String dataId;

    public static final Comparator<ActivityLogEntry> BY_EVENT_TIME =
            (left, right) -> ComparisonChain.start().compare(left.getEventTime(), right.getEventTime()).result();

    public ActivityLogEntry() {
    }

    public ActivityLogEntry(String activityType, String message) {
        this(new Date(), activityType, null, message, null, null);
    }

    public ActivityLogEntry(Date eventTime, String activityType, String username, String message, String targetType, String targetId) {
        this.eventTime = eventTime;
        this.activityType = activityType;
        this.username = username;
        this.message = message;
        this.targetType = targetType;
        this.targetId = targetId;
    }

    public ActivityLogEntry(Date eventTime, String activityType, String username, String message, String targetType, String targetId, String dataId) {
        this.eventTime = eventTime;
        this.activityType = activityType;
        this.username = username;
        this.message = message;
        this.targetType = targetType;
        this.targetId = targetId;
        this.dataId = dataId;
    }

    @PublicApiMember
    public boolean hasUsername() {
        return username != null;
    }

    @PublicApiMember
    public String getUsername() {
        return username;
    }

    @PublicApiMember
    public void setUsername(String username) {
        this.username = username;
    }

    @PublicApiMember
    public String getActivityType() {
        return activityType;
    }

    @PublicApiMember
    public void setMessage(String message) {
        this.message = message;
    }

    @PublicApiMember
    public String getMessage() {
        return message;
    }

    @PublicApiMember
    public Date getEventTime() {
        return eventTime;
    }

    @PublicApiMember
    public void setEventTime(Date eventTime) {
        this.eventTime = eventTime;
    }

    @PublicApiMember
    public void setTargetType(String targetType) {
        this.targetType = targetType;
    }

    @PublicApiMember
    public String getTargetType() {
        return targetType;
    }

    @PublicApiMember
    public void setTargetId(String targetId) {
        this.targetId = targetId;
    }

    @PublicApiMember
    public String getTargetId() {
        return targetId;
    }

    @PublicApiMember
    public String getDataId() {
        return dataId;
    }

    @PublicApiMember
    public void setDataId(final String dataId) {
        this.dataId = dataId;
    }

    @Override
    public String toString() {
        return message;
    }

    private static final Logger logger = LoggerFactory.getLogger(ActivityLogEntry.class);

    public void logEntryToInfo() {
        Map<String, String> oldMDCContextMap = MDC.getCopyOfContextMap();
        if (oldMDCContextMap == null) {
            oldMDCContextMap = new HashMap();
        }
        try {
            MDC.put("username", username);
            MDC.put("activity", activityType);
            if (targetType != null) {
                MDC.put("targetType", targetType);
            }
            if (targetId != null) {
                MDC.put("targetId", targetId);
            }
            logger.info(message);
        } finally {
            MDC.setContextMap(oldMDCContextMap);
        }
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;

        final ActivityLogEntry that = (ActivityLogEntry) o;

        if (username != null ? !username.equals(that.username) : that.username != null) return false;
        if (activityType != null ? !activityType.equals(that.activityType) : that.activityType != null) return false;
        if (message != null ? !message.equals(that.message) : that.message != null) return false;
        if (eventTime != null ? !eventTime.equals(that.eventTime) : that.eventTime != null) return false;
        if (targetType != null ? !targetType.equals(that.targetType) : that.targetType != null) return false;
        if (targetId != null ? !targetId.equals(that.targetId) : that.targetId != null) return false;
        return dataId != null ? dataId.equals(that.dataId) : that.dataId == null;
    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + (username != null ? username.hashCode() : 0);
        result = 31 * result + (activityType != null ? activityType.hashCode() : 0);
        result = 31 * result + (message != null ? message.hashCode() : 0);
        result = 31 * result + (eventTime != null ? eventTime.hashCode() : 0);
        result = 31 * result + (targetType != null ? targetType.hashCode() : 0);
        result = 31 * result + (targetId != null ? targetId.hashCode() : 0);
        result = 31 * result + (dataId != null ? dataId.hashCode() : 0);
        return result;
    }
}
