package com.xebialabs.xlrelease.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.utils.FolderId;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.xebialabs.xlrelease.domain.ScriptHelper.HIDDEN;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static com.xebialabs.xlrelease.variable.VariableHelper.checkVariables;
import static com.xebialabs.xlrelease.variable.VariableHelper.indexByKey;
import static com.xebialabs.xlrelease.variable.VariableHelper.withVariableSyntax;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(versioned = false, virtual = false)
public class ReleaseTrigger extends ScheduledTrigger {
    public static final String SCRIPT_VARS_CATEGORY = "variables";

    @PublicApiMember
    @Property(description = "Name given to releases created by this trigger. This field can contain variables.")
    private String releaseTitle;

    @Property(required = false, category = HIDDEN, description = "Not in Use. Keep it for backwards compatibility.")
    private String executionId;

    @PublicApiMember
    @Property(asContainment = true, required = false, category = HIDDEN)
    protected List<Variable> variables = Lists.newArrayList();

    @Property(required = false, category = HIDDEN)
    private String template;

    @PublicApiMember
    @Property(required = false, category = HIDDEN)
    protected List<String> tags = Lists.newArrayList();

    @PublicApiMember
    @Property(required = false, category = HIDDEN, description = "Folder where releases generated by the trigger will be created.")
    private String releaseFolder;

    @PublicApiMember
    public String getReleaseTitle() {
        return releaseTitle;
    }

    @PublicApiMember
    public void setReleaseTitle(String releaseTitle) {
        this.releaseTitle = releaseTitle;
    }

    @Override
    public List<String> getInternalProperties() {
        ArrayList<String> properties = new ArrayList<>(super.getInternalProperties());
        properties.addAll(this.getScriptVariableNames());
        properties.add("executionId");
        return properties;
    }

    /**
     * This method is kept for backwards compatibility. {@link #getVariables()} method provides richer
     * access to the variable management.
     *
     * @return mapping from variable name to variable value containing only variables with string values.
     * Variable names are in <code>${key}</code> format.
     */
    @PublicApiMember
    public Map<String, String> getTemplateVariables() {
        return VariableHelper.getVariableValuesAsStrings(variables);
    }

    /**
     * This method is kept for backwards compatibility. {@link #getVariables()} method provides richer
     * access to the variable management.
     *
     * @return mapping from variable name to variable value containing only variables with string values.
     * Variable names are in <code>${key}</code> format.
     */
    @PublicApiMember
    public Map<String, String> getTemplatePasswordVariables() {
        return VariableHelper.getPasswordVariableValuesAsStrings(variables);
    }

    @PublicApiMember
    public List<Variable> getVariables() {
        return Lists.newArrayList(variables);
    }

    public void setVariables(final List<Variable> variables) {
        checkVariables(variables);
        this.variables = variables;
    }

    public String getTemplate() {
        return template;
    }

    public void setTemplate(String template) {
        this.template = template;
    }

    public List<String> getTags() {
        return tags;
    }

    public void setTags(List<String> tags) {
        this.tags = tags;
    }

    public String getExecutionId() {
        return executionId;
    }

    public boolean hasExecutionId() {
        return executionId != null;
    }

    @PublicApiMember
    public String getReleaseFolder() { return releaseFolder; }

    @PublicApiMember
    public void setReleaseFolder(String releaseFolder) { this.releaseFolder = releaseFolder; }

    @VisibleForTesting
    public void setExecutionId(String executionId) {
        this.executionId = executionId;
    }

    public void resetExecutionId() {
        executionId = null;
    }

    public void setTriggerStateFromResults(String state) {
        if (state != null && !state.trim().isEmpty()) {
            setTriggerState(state);
        }
    }

    public List<String> getScriptVariableNames() {
        return getType().getDescriptor().getPropertyDescriptors()
                .stream()
                .filter(pd -> SCRIPT_VARS_CATEGORY.equals(pd.getCategory()))
                .map(PropertyDescriptor::getName)
                .collect(toList());
    }

    public void setScriptVariablesFromResults(Map<String, Object> variableValues) {
        for (String scriptVariable : getScriptVariableNames()) {
            setProperty(scriptVariable, variableValues.get(scriptVariable));
        }
    }

    public Map<String, String> getStringScriptVariableValues() {
        return getScriptVariableValues().entrySet()
                .stream()
                .filter(e -> e.getValue() != null)
                .collect(toMap(e -> withVariableSyntax(e.getKey()), e -> e.getValue().toString()));
    }

    public Map<String, Object> getScriptVariableValues() {
        return getScriptVariableNames()
                .stream()
                .collect(toMap(identity(), this::getProperty));
    }

    public Map<String, Object> getTemplateVariableValues(Predicate<Variable> variablePredicate) {
        return this.getVariables().stream()
                .filter(variablePredicate)
                .collect(toMap(Variable::getKey, Variable::getValue));
    }

    public Map<String, Variable> getVariablesByKeys() {
        return indexByKey(this.getVariables());
    }

    @Override
    public String getContainerId() {
        FolderId folder = FolderId.apply(this.getFolderId());
        String containerId = folder.isEmpty() ? this.getTemplate() : folder.absolute();
        return containerId;
    }
}
