package com.xebialabs.xlrelease.domain;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.BOOLEAN;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.DATE;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.ENUM;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.INTEGER;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_OUTPUT;

public class TaskWithPropertiesDefinition extends TaskDefinition {

    private static final String CATEGORY_TRANSITIONAL = "script";

    private static final Predicate<PropertyDescriptor> IS_INPUT_PROPERTY =
            byCategory(CATEGORY_INPUT).and(
                    ofKind(STRING).or(
                            notAPassword().and(
                                    ofKind(BOOLEAN, INTEGER, LIST_OF_STRING, SET_OF_STRING, MAP_STRING_STRING, CI, ENUM, DATE)
                            )
                    )
            );

    private static final Predicate<PropertyDescriptor> IS_TRANSITIONAL_OR_OUTPUT_PROPERTY =
            isVisible().and(
                    ofKind(STRING).or(
                            notAPassword().and(
                                    ofKind(BOOLEAN, INTEGER, LIST_OF_STRING, SET_OF_STRING, MAP_STRING_STRING, DATE))
                    ));

    private static final Predicate<PropertyDescriptor> IS_OUTPUT_PROPERTY =
            byCategory(CATEGORY_OUTPUT).and(IS_TRANSITIONAL_OR_OUTPUT_PROPERTY);

    private static final Predicate<PropertyDescriptor> IS_TRANSITIONAL_PROPERTY =
            byCategory(CATEGORY_TRANSITIONAL).and(IS_TRANSITIONAL_OR_OUTPUT_PROPERTY);

    private final Collection<PropertyDescriptor> inputProperties;
    private final Collection<PropertyDescriptor> outputProperties;
    private final Collection<PropertyDescriptor> transitionalProperties;

    public TaskWithPropertiesDefinition(Type type, boolean allowed) {
        super(type, allowed);

        Collection<PropertyDescriptor> properties = type.getDescriptor().getPropertyDescriptors();
        this.inputProperties = properties.stream().filter(IS_INPUT_PROPERTY).collect(Collectors.toList());
        this.outputProperties = properties.stream().filter(IS_OUTPUT_PROPERTY).collect(Collectors.toList());

        this.transitionalProperties = properties.stream().filter(IS_TRANSITIONAL_PROPERTY).collect(Collectors.toList());
    }

    public TaskWithPropertiesDefinition(Type type) {
        this(type, true);
    }

    public Collection<PropertyDescriptor> getInputProperties() {
        return inputProperties;
    }

    public Collection<PropertyDescriptor> getOutputProperties() {
        return outputProperties;
    }

    Collection<PropertyDescriptor> getTransitionalProperties() {
        return transitionalProperties;
    }

    public Collection<PropertyDescriptor> getTransitionalAndOutputProperties() {
        return Stream.concat(transitionalProperties.stream(), outputProperties.stream()).collect(Collectors.toList());
    }

    public static Predicate<PropertyDescriptor> ofKind(PropertyKind... propertyKinds) {
        return input -> Arrays.asList(propertyKinds).contains(input.getKind());
    }

    private static Predicate<PropertyDescriptor> notAPassword() {
        return input -> !input.isPassword();
    }

    private static Predicate<PropertyDescriptor> byCategory(final String category) {
        return property -> category.equals(property.getCategory());
    }

    private static Predicate<PropertyDescriptor> isVisible() {
        return input -> !input.isHidden();
    }

}
