package com.xebialabs.xlrelease.domain.variables.reference;

import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlrelease.domain.PlanItem;
import com.xebialabs.xlrelease.domain.ReleaseExtension;
import com.xebialabs.xlrelease.domain.ReleaseVisitor;
import com.xebialabs.xlrelease.domain.VisitableItem;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.variables.ValueProviderConfiguration;
import com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType;
import com.xebialabs.xlrelease.repository.CiProperty;

import static com.google.common.collect.Maps.newTreeMap;
import static com.google.common.collect.Sets.newTreeSet;

public class VariableCollectingVisitor implements ReleaseVisitor {
    public static final String VARIABLE_MAPPING_PROPERTY = "variableMapping";

    private final Map<String, VariableReference> variables = newTreeMap();

    public static Set<VariableReference> collectFrom(VisitableItem item) {
        VariableCollectingVisitor visitor = new VariableCollectingVisitor();
        item.accept(visitor);
        return visitor.result();
    }

    public Set<VariableReference> result() {
        return newTreeSet(variables.values());
    }

    @Override
    public void visit(ReleaseExtension extension) {
        extension.getVariableUsages().forEach(this::collectVariables);
        collectVariablesInVariableMappings(extension);
    }

    @Override
    public void visit(final PlanItem item) {
        item.getVariableUsages().forEach(this::collectVariables);
        collectVariablesInVariableMappings(item);
    }

    @Override
    public void visit(final Facet facet) {
        facet.getVariableUsages().forEach(this::collectVariables);
        collectVariablesInVariableMappings(facet);
    }

    @Override
    public void visit(final ValueProviderConfiguration valueProviderConfiguration) {
        collectVariablesInVariableMappings(valueProviderConfiguration);
    }

    private void collectVariables(UsagePoint usagePoint) {
        Map<String, VariableUsageType> variablesAtUsagePoint = usagePoint.collectVariables();

        variablesAtUsagePoint.forEach((variableName, variableType) -> {
            VariableReference variableRef = variables.getOrDefault(variableName, new VariableReference(variableName, variableType));
            variables.putIfAbsent(variableName, variableRef);
            variableRef.addUsagePoint(usagePoint, variableType);
        });
    }

    private void collectVariablesInVariableMappings(final BaseConfigurationItem item) {
        if (item.hasProperty(VARIABLE_MAPPING_PROPERTY)) {
            Map<String, String> variableMapping = item.getProperty(VARIABLE_MAPPING_PROPERTY);
            if (variableMapping != null) {
                variableMapping.forEach((fqPropertyName, variable) -> {
                    Optional<CiProperty> ciProperty = CiProperty.of(item, fqPropertyName);
                    if (variable != null && ciProperty.isPresent()) {
                        collectVariables(new VariableMappingUsagePoint(item, fqPropertyName, ciProperty.get()));
                    }
                });
            }
        }
    }
}
