package com.xebialabs.xlrelease.repository;

import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.domain.variables.Variable;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.xlrelease.variable.VariableHelper.formatVariableIfNeeded;
import static com.xebialabs.xlrelease.variable.VariableHelper.safeReplace;
import static com.xebialabs.xlrelease.variable.VariableHelper.withVariableSyntax;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static java.lang.String.format;

public abstract class CiProperty {
    private static final Pattern PROPERTY_PATTERN = Pattern.compile("\\.");
    private CiProperty wrapped;
    private Property lastProperty;
    private ConfigurationItem parent;
    private boolean exists = true;

    private static Map<PropertyKind, Function<CiProperty, CiProperty>> ciPropertyBuilderPerKind = ImmutableMap.of(
            STRING, StringCiProperty::new,
            LIST_OF_STRING, ListOfStringCiProperty::new,
            SET_OF_STRING, SetOfStringCiProperty::new,
            MAP_STRING_STRING, MapStringStringCiProperty::new
    );

    public static Optional<CiProperty> of(ConfigurationItem rootCi, String fqPropertyExpression) {
        CiProperty ciProperty = new DefaultCiProperty(rootCi, fqPropertyExpression);
        if (ciProperty.exists()) {
            if (ciProperty.isIndexed()) {
                ciProperty = new IndexedCiProperty(ciProperty, ciProperty.lastProperty.index);
            }
            if (ciPropertyBuilderPerKind.containsKey(ciProperty.getKind())) {
                ciProperty = ciPropertyBuilderPerKind.get(ciProperty.getKind()).apply(ciProperty);
            }
            return Optional.of(ciProperty);
        }
        return Optional.empty();
    }

    protected CiProperty(CiProperty wrapped) {
        this.parent = wrapped.parent;
        this.exists = wrapped.exists;
        this.lastProperty = wrapped.lastProperty;
        this.wrapped = wrapped;
    }

    protected CiProperty(ConfigurationItem rootCi, String fqPropertyExpression) {
        Optional<Object> newParent = Optional.ofNullable(rootCi);
        for (String propertyExpression : PROPERTY_PATTERN.split(fqPropertyExpression)) {
            if (newParent.isPresent()) {
                this.parent = (ConfigurationItem) newParent.get();
                this.lastProperty = Property.of(propertyExpression);
                if (parent.hasProperty(lastProperty.propertyName)) {
                    newParent = Optional.ofNullable(getDescriptor().get(getParentCi()));
                    if (lastProperty.isIndexed()) {
                        Collection collection = (Collection) newParent.orElse(Collections.emptyList());
                        newParent = collection.stream().skip(lastProperty.index).findFirst();
                        if (!newParent.isPresent()) {
                            this.exists = false;
                            break;
                        }
                    }
                } else {
                    this.exists = false;
                    break;
                }
            } else {
                this.exists = false;
                break;
            }
        }
    }

    public String getPropertyName() {
        return this.lastProperty.propertyName;
    }

    public <T> T getValue() {
        return wrapped.getValue();
    }

    public void setValue(Object value) {
        wrapped.setValue(value);
    }

    public ConfigurationItem getParentCi() {
        return this.parent;
    }

    public PropertyDescriptor getDescriptor() {
        return parent.getType().getDescriptor().getPropertyDescriptor(lastProperty.propertyName);
    }

    public PropertyKind getKind() {
        return getDescriptor().getKind();
    }

    public String getCategory() {
        return getDescriptor().getCategory();
    }

    public boolean isPassword() {
        return getDescriptor().isPassword();
    }

    public boolean isIndexed() {
        return lastProperty.isIndexed();
    }

    public boolean exists() {
        return exists;
    }

    public void replaceInValue(final Variable variable, final VariableOrValue replacement) {
        String key = withVariableSyntax(variable.getKey());
        String newValue;
        if (replacement.getVariable() != null) {
            newValue = formatVariableIfNeeded(replacement.getVariable());
        } else {
            Variable dummy = variable.getType().getDescriptor().newInstance("dummy");
            dummy.setUntypedValue(replacement.getValue());
            newValue = dummy.getValueAsString();
        }
        replaceInStrings(key, newValue);
    }

    protected abstract void replaceInStrings(String variableKey, String replacement);

    private static class Property {
        private String INDEXED_PROPERTY_PATTERN = "^([^\\.\\[\\]]+)\\[(\\d+)\\]$";
        String propertyName;
        Integer index;

        public static Property of(String expression) {
            return new Property(expression);
        }

        public boolean isIndexed() {
            return null != index;
        }

        private Property(String expression) {
            if (isIndexed(expression)) {
                Matcher matcher = Pattern.compile(INDEXED_PROPERTY_PATTERN).matcher(expression);
                if (matcher.matches()) {
                    propertyName = matcher.group(1);
                    index = Integer.valueOf(matcher.group(2));
                } else {
                    throw new IllegalStateException(format("Expression '%s' is not indexed property.", expression));
                }
            } else {
                propertyName = expression;
            }
        }

        private boolean isIndexed(final String expression) {
            return expression.matches(INDEXED_PROPERTY_PATTERN);
        }
    }
}

class DefaultCiProperty extends CiProperty {

    protected DefaultCiProperty(final ConfigurationItem rootCi, final String fqPropertyName) {
        super(rootCi, fqPropertyName);
    }

    @Override
    protected void replaceInStrings(final String variableKey, final String replacement) {
        // noop, as property value does not contain strings, so cannot mention variableKey
    }

    public <T> T getValue() {
        Object value = getDescriptor().get(getParentCi());
        //noinspection unchecked
        return (T) value;
    }

    public void setValue(Object value) {
        getParentCi().setProperty(getPropertyName(), value);
    }

}

class IndexedCiProperty extends CiProperty {

    private Integer index;

    protected IndexedCiProperty(final CiProperty wrapped, Integer index) {
        super(wrapped);
        this.index = index;
    }

    @Override
    protected void replaceInStrings(final String variableKey, final String replacement) {
        // noop, as this merely abstracts getValue and setValue
    }

    @Override
    public void setValue(final Object newValue) {
        Collection collection = (Collection) getDescriptor().get(getParentCi());
        Object[] elements = collection.stream().toArray();
        elements[index] = newValue;
        List<Object> value = Arrays.asList(elements);
        super.setValue(value);
    }

    @Override
    public <T> T getValue() {
        Collection collection = super.getValue();
        Object value = collection.stream().skip(index).findFirst().get();
        //noinspection unchecked
        return (T) value;
    }
}

class StringCiProperty extends CiProperty {

    protected StringCiProperty(final CiProperty wrapped) {
        super(wrapped);
    }

    @Override
    protected void replaceInStrings(final String variableKey, final String replacement) {
        setValue(safeReplace(getValue(), variableKey, replacement));
    }
}

class ListOfStringCiProperty extends CiProperty {
    public ListOfStringCiProperty(final CiProperty wrapped) {
        super(wrapped);
    }

    @Override
    protected void replaceInStrings(final String variableKey, final String replacement) {
        if (isIndexed()) {
            String replacedElement = safeReplace(getValue(), variableKey, replacement);
            setValue(replacedElement);
        } else {
            List<String> list = getValue();
            List<String> listReplaced = list.stream()
                    .map(s -> safeReplace(s, variableKey, replacement))
                    .collect(Collectors.toList());
            setValue(listReplaced);
        }
    }
}

class SetOfStringCiProperty extends CiProperty {
    public SetOfStringCiProperty(final CiProperty wrapped) {
        super(wrapped);
    }

    @Override
    protected void replaceInStrings(final String variableKey, final String replacement) {
        Set<String> set = getValue();
        Set<String> setReplaced = set.stream()
                .map(s -> safeReplace(s, variableKey, replacement))
                .collect(Collectors.toSet());
        setValue(setReplaced);
    }
}

class MapStringStringCiProperty extends CiProperty {
    public MapStringStringCiProperty(final CiProperty wrapped) {
        super(wrapped);
    }

    @Override
    protected void replaceInStrings(final String variableKey, final String replacement) {
        Map<String, String> value = getValue();
        value = value.entrySet().stream().collect(toMap(
                entry -> safeReplace(entry.getKey(), variableKey, replacement),
                entry -> safeReplace(entry.getValue(), variableKey, replacement))
        );
        setValue(value);
    }
}
