package com.xebialabs.xlrelease.domain.delivery;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.internal.InternalMetadata;
import com.xebialabs.xlrelease.domain.CiWithInternalMetadata;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Ids;

import static com.xebialabs.xlrelease.domain.delivery.DeliveryStatus.COMPLETED;
import static java.util.stream.Collectors.toSet;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(root = Metadata.ConfigurationItemRoot.BY_ROOT_NAME, rootName = Delivery.DELIVERY_ROOT, versioned = false)
public class Delivery extends BaseConfigurationItem implements CiWithInternalMetadata {
    public static final String DELIVERY_ROOT = Ids.DELIVERIES_ROOT;

    /**
     * An additional derived Release metadata that is used in our UI. Does not
     * belong to public API and can be changed. See implementations of com.xebialabs.xlrelease.api.internal.InternalMetadataDecorator
     * for usages.
     */
    private Map<String, InternalMetadata> $metadata = new LinkedHashMap<>();

    @Property(description = "Symbolic name for the release delivery.")
    private String title;

    @Property(required = false, description = "Description for the delivery pattern.")
    private String description;

    @Property(description = "The state the release delivery is in.")
    private DeliveryStatus status = DeliveryStatus.IN_PROGRESS;

    @Property(required = false, description = "The expected start date.")
    private Date startDate;

    @Property(required = false, description = "The expected end date.")
    private Date endDate;

    @Property(required = false, description = "The time that the release delivery is supposed to take to complete, in hours.")
    private Integer plannedDuration = 0;

    @Property(required = false, description = "List of releases contained by this release delivery.")
    private Set<String> releaseIds = new HashSet<>();

    @Property(required = false, isTransient = true, description = "Folder that the release delivery belongs to.")
    private String folderId;

    @Property(required = false, description = "The ID of the pattern that created this delivery.")
    private String originPatternId;

    @Property(asContainment = true)
    private List<Stage> stages = new ArrayList<>();

    @Property(asContainment = true)
    private List<TrackedItem> trackedItems = new ArrayList<>();

    @Property(asContainment = true)
    private List<Subscriber> subscribers = new ArrayList<>();

    @Property(required = false, description = "Complete delivery when all items are completed, skipped or de-scoped in all stages")
    private boolean autoComplete;
    // ----

    public boolean isTemplate() {
        return this.status != null && this.status.isTemplate();
    }

    public void updateDuration() {
        LocalDateTime localScheduledStartDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(startDate.getTime()), ZoneId.systemDefault());
        LocalDateTime localDueDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(endDate.getTime()), ZoneId.systemDefault());
        int newDuration = Math.toIntExact(localScheduledStartDate.until(localDueDate, ChronoUnit.HOURS));
        setPlannedDuration(newDuration);
    }

    // ITEMS

    public Optional<TrackedItem> findItemByIdOrTitle(String idOrTitle) {
        return this.getTrackedItems().stream()
                .filter(item -> Ids.getName(item.getId()).equals(Ids.getName(idOrTitle)) || item.getTitle().equalsIgnoreCase(idOrTitle))
                .findFirst();
    }

    public TrackedItem getItemByIdOrTitle(String idOrTitle) {
        return findItemByIdOrTitle(idOrTitle).orElseThrow(() -> new LogFriendlyNotFoundException("Tracked item '%s' does not exist in delivery '%s'", idOrTitle, title));
    }

    // STAGES

    public void addStage(Stage stage) {
        this.addStage(stage, this.stages.size());
    }

    public void addStage(Stage stage, int position) {
        this.stages.add(position, stage);
    }

    public void removeStage(Stage stage) {
        this.stages.remove(stage);
    }

    public boolean isLastStage(Stage stage) {
        return this.getStages().indexOf(stage) == (this.getStages().size() - 1);
    }

    public Optional<Stage> findFirstOpenStage() {
        return this.getStages().stream().filter(Stage::isOpen).findFirst();
    }

    public Optional<Stage> findPreviousStage(Stage stage) {
        int index = this.getStages().indexOf(stage);
        if (index <= 0) return Optional.empty();
        else return Optional.of(this.getStages().get(index - 1));
    }

    public Optional<Stage> findNextStage(Stage stage) {
        int index = this.getStages().indexOf(stage);
        if (index < 0 || index + 1 == this.getStages().size()) return Optional.empty();
        else return Optional.of(this.getStages().get(index + 1));
    }

    public Optional<Stage> findStageByIdOrTitle(String idOrTitle) {
        final String idName = Ids.getName(idOrTitle);
        return this.getStages().stream()
                .filter(stage ->
                        Ids.getName(stage.getId()).equals(idName) || stage.getTitle().equalsIgnoreCase(idOrTitle))
                .findFirst();
    }

    public Stage getStageByIdOrTitle(String idOrTitle) {
        return findStageByIdOrTitle(idOrTitle).orElseThrow(() ->
                new LogFriendlyNotFoundException("Stage '%s' does not exist in delivery '%s'", idOrTitle, title));
    }

    @PublicApiMember
    public List<Transition> getTransitions() {
        return this.stages.stream()
                .filter(stage -> stage.getTransition() != null)
                .map(Stage::getTransition)
                .collect(Collectors.toList());
    }

    public Optional<Transition> findTransitionByIdOrTitle(String transitionIdOrTitle) {
        return this.stages.stream()
                .filter(s -> s.getTransition() != null &&
                        (Ids.getName(s.getTransition().getId()).equals(Ids.getName(transitionIdOrTitle)) ||
                                s.getTransition().getTitle().equals(transitionIdOrTitle)))
                .map(Stage::getTransition)
                .findFirst();
    }

    public Transition getTransitionByIdOrTitle(String transitionIdOrTitle) {
        return findTransitionByIdOrTitle(transitionIdOrTitle).orElseThrow(() ->
                new LogFriendlyNotFoundException("Transition '%s' does not exist in delivery '%s'", transitionIdOrTitle, title));
    }

    public Stage getStageByTransition(Transition transition) {
        return getStageByTransitionId(transition.getId());
    }

    public Stage getStageByTransitionId(String transitionId) {
        Transition transition = getTransitionByIdOrTitle(transitionId);
        return transition.getStage();
    }

    public List<Stage> getStagesBeforeFirstOpenTransition() {
        List<Stage> stagesBefore = new ArrayList<>();
        for (Stage stage : getStages()) {
            stagesBefore.add(stage);
            if (stage.isOpen() && stage.getTransition() != null) {
                break;
            }
        }
        return stagesBefore;
    }

    public List<Stage> getStagesBefore(Stage stage) {
        return stages.subList(0, stages.indexOf(stage));
    }

    public List<Stage> getStageGroupAfterTransition(Transition transition) {
        List<Stage> stageGroup = new ArrayList<>();
        Stage transitionStage = getStageByTransition(transition);
        int idx = stages.indexOf(transitionStage);
        if (idx != -1) {
            idx++;
            while (idx < stages.size()) {
                Stage s = stages.get(idx);
                stageGroup.add(s);
                if (s.getTransition() != null) {
                    break;
                }
                idx++;
            }
        }
        return stageGroup;
    }

    public List<Stage> getStageGroupOfStage(Stage stage) {
        List<Stage> stageGroup = new ArrayList<>();
        int index = stages.indexOf(stage);
        if (index != -1) {
            // find from left side and right side
            // left
            List<Stage> leftList = new ArrayList<>();
            int lidx = index - 1;
            while (lidx >= 0) {
                Stage s = stages.get(lidx);
                if (s.getTransition() == null) {
                    leftList.add(s);
                } else {
                    break;
                }
                lidx--;
            }
            // right
            List<Stage> rightList = new ArrayList<>();
            int ridx = index;
            while (ridx < stages.size()) {
                Stage s = stages.get(ridx);
                rightList.add(s);
                if (s.getTransition() != null) {
                    break;
                }
                ridx++;
            }
            Collections.reverse(leftList);
            stageGroup.addAll(leftList);
            stageGroup.addAll(rightList);
        }
        return stageGroup;
    }

    public <T extends Subscriber> List<T> getSubscribersOfType(Class<T> subscriberClass) {
        return getSubscribers().stream()
                .filter(subscriber -> subscriber.getType().instanceOf(Type.valueOf(subscriberClass)))
                .map(subscriberClass::cast)
                .collect(Collectors.toList());
    }

    public void addSubscriber(final Subscriber subscriber) {
        this.subscribers.add(subscriber);
    }

    @PublicApiMember
    public Optional<Subscriber> findSubscriberBySourceId(String sourceId) {
        return getSubscribers().stream().filter(subscriber -> subscriber.getSourceId().equals(sourceId)).findFirst();
    }

    // ----
    @PublicApiMember
    public String getTitle() {
        return title;
    }

    @PublicApiMember
    public void setTitle(final String title) {
        this.title = title;
    }

    @PublicApiMember
    public String getDescription() {
        return description;
    }

    @PublicApiMember
    public void setDescription(String description) {
        this.description = description;
    }

    @PublicApiMember
    public DeliveryStatus getStatus() {
        return status;
    }

    @PublicApiMember
    public void setStatus(final DeliveryStatus status) {
        this.status = status;
    }

    @PublicApiMember
    public Date getStartDate() {
        return startDate;
    }

    @PublicApiMember
    public void setStartDate(final Date startDate) {
        this.startDate = startDate;
    }

    @PublicApiMember
    public Date getEndDate() {
        return endDate;
    }

    @PublicApiMember
    public void setEndDate(final Date endDate) {
        this.endDate = endDate;
    }

    @PublicApiMember
    public Set<String> getReleaseIds() {
        return releaseIds;
    }

    @PublicApiMember
    public void setReleaseIds(final Set<String> releaseIds) {
        this.releaseIds = releaseIds;
    }

    public void removeReleaseIds(List<String> idsToRemove) {
        //quite messy so need to figure out a different way to accept the different ways an id is passed in
        List<String> strippedIds = idsToRemove.stream().map(Ids::getName).collect(Collectors.toList());
        releaseIds = releaseIds.stream().filter(id -> !strippedIds.contains(Ids.getName(id))).collect(toSet());
    }

    public boolean isUpdatable() {
        return !COMPLETED.equals(status);
    }

    @PublicApiMember
    public String getFolderId() {
        return folderId;
    }

    @PublicApiMember
    public void setFolderId(final String folderId) {
        this.folderId = folderId;
    }

    public String getOriginPatternId() {
        return originPatternId;
    }

    public void setOriginPatternId(final String originPatternId) {
        this.originPatternId = originPatternId;
    }

    @Override
    public Map<String, InternalMetadata> get$metadata() {
        return $metadata;
    }

    @PublicApiMember
    public List<TrackedItem> getTrackedItems() {
        return trackedItems;
    }

    @PublicApiMember
    public void setTrackedItems(List<TrackedItem> trackedItems) {
        this.trackedItems = trackedItems;
    }

    @PublicApiMember
    public List<Stage> getStages() {
        return stages;
    }

    @PublicApiMember
    public void setStages(final List<Stage> stages) {
        this.stages = stages;
    }

    public void addReleaseId(final String releaseId) {
        this.releaseIds.add(releaseId);
    }

    public void addReleaseIds(final Set<String> releaseIds) {
        this.releaseIds.addAll(releaseIds);
    }

    public void addTrackedItem(final TrackedItem item) {
        this.trackedItems.add(item);
    }

    public void removeTrackedItem(String itemId) {
        trackedItems.removeIf(item -> Ids.getName(item.getId()).equals(Ids.getName(itemId)));
    }

    @PublicApiMember
    public Integer getPlannedDuration() {
        return plannedDuration;
    }

    public void setPlannedDuration(Integer plannedDuration) {
        this.plannedDuration = plannedDuration;
    }

    public List<Subscriber> getSubscribers() {
        return subscribers;
    }

    public void setSubscribers(final List<Subscriber> subscribers) {
        this.subscribers = subscribers;
    }

    public void computeReleasesFromTrackedItems() {
        Set<String> updatedReleaseIds = new HashSet<>();
        trackedItems.forEach(item -> updatedReleaseIds.addAll(item.getReleaseIds()));
        setReleaseIds(updatedReleaseIds);
    }

    public boolean isAutoComplete() {
        return autoComplete;
    }

    public void setAutoComplete(boolean autoComplete) {
        this.autoComplete = autoComplete;
    }
}
