package com.xebialabs.xlrelease.domain;

import java.io.Serializable;
import java.util.*;
import org.joda.time.DateTime;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseOperation;
import com.xebialabs.xlrelease.service.PostAction;
import com.xebialabs.xlrelease.user.User;

import scala.Tuple3;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;

public class Changes {

    public Changes() {
        this.lastComment = DateTime.now();
    }

    private DateTime lastComment;

    private Set<ConfigurationItem> updatedItems = newLinkedHashSet();

    private Multimap<Task, Tuple3<User, String, DateTime>> commentsByTask = ArrayListMultimap.create();
    private Multimap<Task, Attachment> attachmentsByTaskId = ArrayListMultimap.create();
    private Multimap<Task, Attachment> linkedAttachments = ArrayListMultimap.create();

    private Set<String> removedIds = newLinkedHashSet();

    private List<PostAction> postActions = newArrayList();

    private List<Task> tasksToBackup = newArrayList();

    private List<XLReleaseOperation> operations = new ArrayList<>();

    private Set<String> variablesUsed = new HashSet<>();

    public void addAll(Changes changes) {
        this.updatedItems.addAll(changes.getUpdatedItems());
        commentsByTask.putAll(changes.getCommentsByTask());
        removedIds.addAll(changes.getRemovedIds());
        postActions.addAll(changes.getPostActions());
        tasksToBackup.addAll(changes.getTasksToBackup());
        operations.addAll(changes.getOperations());
    }

    public void update(ConfigurationItem item) {
        updatedItems.add(item);
    }

    public void updateAll(Collection<? extends ConfigurationItem> items) {
        updatedItems.addAll(items);
    }

    public void addComment(Task task, String comment) {
        addComment(task, null, comment);
    }

    public void addComment(Task task, User user, String comment) {
        if (!isBlank(comment)) {
            DateTime now = DateTime.now();
            if (now.isAfter(lastComment)) {
                lastComment = now;
            } else {
                lastComment = lastComment.plusMillis(1);
            }
            commentsByTask.put(task, new Tuple3<>(user, comment, lastComment));
        }
    }

    public void addPostAction(PostAction postAction) {
        postActions.add(postAction);
    }

    public void remove(String id) {
        this.removedIds.add(id);
    }

    public boolean hasUpdatedItems() {
        return !updatedItems.isEmpty();
    }

    public Set<ConfigurationItem> getUpdatedItems() {
        return updatedItems;
    }

    public ConfigurationItem[] toUpdatedItemsArray() {
        return updatedItems.toArray(new ConfigurationItem[updatedItems.size()]);
    }

    public Multimap<Task, Tuple3<User, String, DateTime>> getCommentsByTask() {
        return commentsByTask;
    }

    public Multimap<Task, Attachment> getAttachmentsByTask() {
        return attachmentsByTaskId;
    }

    public Set<String> getRemovedIds() {
        return removedIds;
    }

    public List<PostAction> getPostActions() {
        return postActions;
    }

    public List<XLReleaseOperation> getOperations() {
        return operations;
    }

    public void addOperation(XLReleaseOperation operation) {
        operations.add(operation);
    }

    public void addOperations(List<XLReleaseOperation> operations) {
        this.operations.addAll(operations);
    }

    public void addAttachment(final Task task, Attachment attachment) {
        attachmentsByTaskId.put(task, attachment);
    }

    public List<Task> getTasksToBackup() {
        return tasksToBackup;
    }

    public void addTaskToBackup(Task task) {
        tasksToBackup.add(task);
    }

	public void addVariablesUsed(Collection<String> variableNamesUsed) {
		variablesUsed.addAll(variableNamesUsed);
	}

	public Set<String> getVariablesUsed() {
		return Collections.unmodifiableSet(variablesUsed);
	}

	@SafeVarargs
    public final void addOperationBefore(XLReleaseOperation operation, Class<? extends XLReleaseOperation>... operationBeforeClazzez) {
        for (int i = 0; i < operations.size(); i++) {
            final XLReleaseOperation existingOperation = operations.get(i);
            if (Arrays.stream(operationBeforeClazzez).anyMatch(clazz -> clazz.isInstance(existingOperation))) {
                operations.add(i, operation);
                return;
            }
        }
        operations.add(operation);
    }

    public void linkScriptOutputLog(final Task task, final String artifactId) {
        Release release = task.getRelease();
        Attachment attachment = new Attachment();
        attachment.setContentType("text/plain");
        attachment.setId(artifactId);
        release.getAttachments().add(attachment);
        task.getAttachments().add(attachment);
        update(task.getRelease());
        update(task);
        linkedAttachments.put(task, attachment);
    }

    public Multimap<Task, Attachment> getLinkedAttachments() {
        return linkedAttachments;
    }

    public static class VariablesChanges implements Serializable {
        public static final VariablesChanges EMPTY = new VariablesChanges();

        private final List<Variable> createdVariables = new ArrayList<>();
        private final List<Variable> updatedVariables = new ArrayList<>();
        private final List<Variable> deletedVariables = new ArrayList<>();
        private final List<XLReleaseOperation> operations = new ArrayList<>();

        public List<Variable> getCreatedVariables() {
            return createdVariables;
        }

        public List<Variable> getUpdatedVariables() {
            return updatedVariables;
        }

        public List<Variable> getDeletedVariables() {
            return deletedVariables;
        }

        public List<XLReleaseOperation> getOperations() {
            return operations;
        }
    }
}
