package com.xebialabs.xlrelease.domain;

import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;
import com.xebialabs.xlrelease.utils.PythonScriptCiHelper;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.xlrelease.domain.ContainerTaskDefinition.ofKind;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAll;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;
import static java.util.stream.Collectors.toList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Generic containerized task", versioned = false, virtual = true)
public class ContainerTask extends Task implements RemoteExecution {

    @Property(category = CATEGORY_INPUT, label = "Capabilities", description = "Route jobs to particular types of remote runners based on capabilities", defaultValue = "remote")
    private Set<String> capabilities = new HashSet<>();

    @Property(required = false, description = "Status line of the task that will appear in UI.")
    private String statusLine;

    @Property(required = false, hidden = true, defaultValue = "", description = "Icon location for the task")
    private String iconLocation;

    @Property(required = false, hidden = true, defaultValue = "container-icon", description = "Icon class for the task")
    private String iconClass;

    @Property(required = false, hidden = true, defaultValue = "#3D6C9E", description = "Task color")
    private String taskColor;

    @Property(required = false, hidden = true, defaultValue = "The output of the task can be stored in release variables; specify the variables to be used (optional)", description = "Help text below output properties section title")
    private String outputVarHelpText;

    @Property(required = false, hidden = true, defaultValue = "true", description = "Is task automated?")
    private boolean automated;

    @Property(required = false, hidden = true, defaultValue = "false")
    private boolean sendNotificationWhenStarted;

    @Property(hidden = true, description = "Container image name for the task", isTransient = true)
    private String image;

    @Property(required = false, hidden = true, defaultValue = "30", description = "Timeout in seconds for abort operation")
    private Integer abortTimeout;

    @Property(required = false, hidden = true, defaultValue = "12", description = "Max retry attempts in case of connection failure")
    private Integer maxRetryAttempts;

    @Property(required = false, hidden = true, defaultValue = "5", description = "Delay in seconds between retry attempts in case of connection failure")
    private Integer retryDelay;

    @Property(required = false, description = "Keep the previous output properties on retry")
    private boolean keepPreviousOutputPropertiesOnRetry;

    public Set<String> getCapabilities() {
        return capabilities;
    }

    public void setCapabilities(final Set<String> capabilities) {
        this.capabilities = capabilities;
    }

    @Override
    public Changes retry(final String targetId) {
        getTransitionalAndOutputProperties()
                .stream()
                .filter(pd -> !isKeepPreviousOutputPropertiesOnRetry() || PythonScriptCiHelper.isEmpty(getProperty(pd.getName())))
                .forEach(
                        propertyDescriptor -> propertyDescriptor.set(this, propertyDescriptor.getDefaultValue())
                );
        statusLine = null;
        return super.retry(targetId);
    }

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        Changes changes = super.execute(targetId, operation);

        if (getStatus() != TaskStatus.IN_PROGRESS) {
            return changes;
        }

        generateExecutionId();
        changes.update(this);

        changes.addPostAction(new ExecuteTaskAction(this));

        return changes;
    }

    @Override
    protected boolean shouldFreezeVariableMapping(final CiProperty property) {
        return !CATEGORY_OUTPUT.equals(property.getDescriptor().getCategory());
    }

    @Override
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables, Map<String, String> passwordVariables,
                                                     Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = newHashSet();
        List<PropertyDescriptor> propertiesWithVariables = getInputProperties().stream()
                .filter(ofKind(STRING, LIST_OF_STRING, SET_OF_STRING, MAP_STRING_STRING)).collect(toList());
        final Set<String> variablesInNonInterpolatableValues = getRelease().getVariablesKeysInNonInterpolatableVariableValues()
                .stream().map(v -> VariableHelper.withVariableSyntax(v)).collect(Collectors.toSet());
        for (PropertyDescriptor propertyDescriptor : propertiesWithVariables) {
            final Object value = this.getProperty(propertyDescriptor.getName());
            Object newValue;
            if (propertyDescriptor.isPassword()) {
                newValue = replaceAll(value, passwordVariables, unresolvedVariables, freezeEvenIfUnresolved);
            } else {
                newValue = replaceAllWithInterpolation(value, variables, unresolvedVariables, freezeEvenIfUnresolved);
            }
            this.setProperty(propertyDescriptor.getName(), newValue);
            if (value != null && !value.equals(newValue)) {
                //something got replaced so can now see what variables were used.
                final Set<String> variablesUsed = VariableHelper.collectVariables(value);
                variablesUsed.removeAll(variablesInNonInterpolatableValues);
                changes.addVariablesUsed(variablesUsed);
            }
        }
        if (!propertiesWithVariables.isEmpty()) {
            changes.update(this);
        }
        return unresolvedVariables;
    }


    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        usagePoints.addAll(getInputProperties().stream().map(pd -> new PropertyUsagePoint(this, pd.getName())).collect(toList()));
        usagePoints.addAll(getOutputProperties().stream().map(pd -> new PropertyUsagePoint(this, pd.getName())).collect(toList()));
        return usagePoints;
    }

    public String getIconLocation() {
        return iconLocation;
    }

    public String getOutputVarHelpText() {
        return outputVarHelpText;
    }

    public String getTaskColor() {
        return taskColor;
    }

    public String getIconClass() {
        return iconClass;
    }

    public Integer getAbortTimeout() {
        return abortTimeout;
    }

    public Integer getMaxRetryAttempts() {
        return maxRetryAttempts;
    }

    public Integer getRetryDelay() {
        return retryDelay;
    }

    public String getImage() {
        return image;
    }

    @Override
    public boolean hasAbortScript() {
        return true;
    }

    public String getStatusLine() {
        return statusLine;
    }

    public void setStatusLine(final String statusLine) {
        this.statusLine = statusLine;
    }

    public Collection<PropertyDescriptor> getInputProperties() {
        ContainerTaskDefinition taskDefinition = new ContainerTaskDefinition(getType(), true);
        return taskDefinition.getInputProperties();
    }

    private Collection<PropertyDescriptor> getOutputProperties() {
        ContainerTaskDefinition taskDefinition = new ContainerTaskDefinition(getType(), true);
        return taskDefinition.getOutputProperties();
    }

    @PublicApiMember
    public boolean isKeepPreviousOutputPropertiesOnRetry() {
        return keepPreviousOutputPropertiesOnRetry;
    }

    @PublicApiMember
    public void setKeepPreviousOutputPropertiesOnRetry(final boolean keepPreviousOutputPropertiesOnRetry) {
        this.keepPreviousOutputPropertiesOnRetry = keepPreviousOutputPropertiesOnRetry;
    }

    public Collection<PropertyDescriptor> getTransitionalAndOutputProperties() {
        return new ContainerTaskDefinition(getType(), true).getTransitionalAndOutputProperties();
    }

    public boolean isUnknown() {
        return getType().equals(UNKNOWN_TASK_TYPE);
    }
}
