package com.xebialabs.xlrelease.domain.events

import com.xebialabs.xlrelease.domain.{Release, Team}

import java.util.{List => JList}

//
// Teams
//

/**
  * Common interface for domain events related to comments modification.
  */
sealed trait TeamEvent extends XLReleaseEvent

/**
  * Event published when a team is created.
  * @param containerId the container id to which the team belongs.
  * @param team the created team.
  */
case class TeamCreatedEvent(containerId: String, team: Team) extends TeamEvent

object TeamCreatedEvent {
  def apply(release: Release, team: Team): TeamCreatedEvent = new TeamCreatedEvent(release.getId, team)
}

/**
  * Event published when a team's members or roles are updated.
  *
  * @param containerId the container id to which the team belongs.
  * @param original    a copy of the original team before the update.
  *                    You can use [[original]] to see which members or roles were added or removed.
  * @param updated the updated team.
  */
case class TeamUpdatedEvent(containerId: String, original: Team, updated: Team) extends TeamEvent

object TeamUpdatedEvent {
  def apply(release: Release, original: Team, updated: Team): TeamUpdatedEvent = new TeamUpdatedEvent(release.getId, original, updated)
}

/**
  * Event published when a team is deleted.
  *
  * @param containerId the container id to which the team belongs.
  * @param team the deleted team.
  */
case class TeamDeletedEvent(containerId: String, team: Team) extends TeamEvent

object TeamDeletedEvent {
  def apply(release: Release, team: Team): TeamDeletedEvent = new TeamDeletedEvent(release.getId, team)
}

/**
  * Event published when teams of a template are merged with teams of a folder.
  * This can happen when a template is moved into a folder.
  *
  * @param templateId the template ID which teams were merged.
  * @param folderTitle the folder title in which the teams were merged.
  */
case class TeamsMergedEvent(templateId: String, folderTitle: String) extends TeamEvent

/**
  * Event published when teams of a template are removed when the template is moved to a folder.
  * @param templateId ID of the template which was moved.
  * @param folderTitle Title of the folder to which the template was moved.
  */
case class TeamsRemovedInTemplateEvent(templateId: String, folderTitle: String) extends TeamEvent

//
// Permissions
//

/**
  * Common interface for domain events related to permission changes.
  */
sealed trait PermissionsEvent extends XLReleaseEvent

/**
  * Event published when permissions of a number of teams were updated in a release or a template.
  * @param teams the list of teams with their permissions after the update.
  */
case class PermissionsUpdatedEvent(teams: JList[Team]) extends PermissionsEvent

case class TeamsUpdatedEvent(containerId: String) extends XLReleaseEvent

case class GlobalRolesOrPermissionsUpdatedEvent() extends XLReleaseEvent