package com.xebialabs.xlrelease.risk.domain.riskassessors;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.risk.domain.RiskAssessment;
import com.xebialabs.xlrelease.risk.domain.RiskProfile;

import static java.lang.String.format;

@PublicApiRef
@Metadata(label = "TaskRetriesAssessor", versioned = false)
public class TaskRetriesRiskAssessor extends RiskAssessor {

    public static final String TASK_RETRIES_HEADLINE = "Release contains failed tasks";
    private static final String TASK_RETRIES_MESSAGE = "%s has failed %d %s";

    private String suffix(int failuresCount) {
        return failuresCount == 1 ? "time" : "times";
    }

    @Property(description = "Minimum amount of retries to have to score")
    public int minRetries;

    @Property(description = "Maximum amount of retries to have to score")
    public int maxRetries;

    @Override
    public RiskAssessment execute(final Release release, final RiskProfile riskProfile) {
        List<Task> failedActiveTasks = release.getActiveTasks()
                .stream()
                .filter(task -> task.isFailed() || task.isFailing())
                .collect(Collectors.toList());

        int totalFailedCount = failedActiveTasks.stream()
                .mapToInt(Task::getFailuresCount)
                .sum();

        RiskAssessment assessment = new RiskAssessment();
        assessment.setRiskAssessorId(this.getId());

        if (!failedActiveTasks.isEmpty() &&
            totalFailedCount >= minRetries && totalFailedCount <= maxRetries) {

            List<String> messages = new ArrayList<>();
            failedActiveTasks.forEach(task -> {
                messages.add(format(TASK_RETRIES_MESSAGE, task.getTitle(), task.getFailuresCount(), suffix(task.getFailuresCount())));
            });

            assessment.setScore(riskProfile.getValueFor(getType()));
            assessment.setMessages(messages);
            assessment.setHeadline(TASK_RETRIES_HEADLINE);
        } else {
            assessment.setScore(OK_SCORE);
            assessment.setHeadline(OK_HEADLINE);
        }

        return assessment;
    }

    public int getMinRetries() {
        return minRetries;
    }

    public void setMinRetries(final int minRetries) {
        this.minRetries = minRetries;
    }

    public int getMaxRetries() {
        return maxRetries;
    }

    public void setMaxRetries(final int maxRetries) {
        this.maxRetries = maxRetries;
    }
}
