package com.xebialabs.xlrelease.domain;

import java.util.*;

import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.status.PhaseStatus;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.*;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.user.User;

import static com.xebialabs.xlrelease.domain.status.PhaseStatus.ABORTED;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.COMPLETED;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.FAILED;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.FAILING;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.PLANNED;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.SKIPPED;
import static com.xebialabs.xlrelease.variable.VariableHelper.*;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;


@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(description = "A phase in a release that contains tasks.", versioned = false)
public class Phase extends PlanItem implements TaskContainer {

    private static final Logger logger = LoggerFactory.getLogger(Phase.class);

    private Integer releaseUid;

    @Property(asContainment = true, required = false, description = "The list of tasks in this phase.")
    private List<Task> tasks = new ArrayList<>();

    @Property(asContainment = true, description = "The release this phase belongs to.")
    private Release release;

    @Property(description = "The state the phase is in.")
    protected PhaseStatus status;

    @Property(description = "The color of the phase top bar in the UI. Format: #(hex value); for example '#3D6C9E'")
    protected String color;

    @Property(required = false, description = "If given, then this phase has been copied as a part of restart phase operation based on this id")
    protected String originId;

    @Override
    @PublicApiMember
    public List<Task> getTasks() {
        return tasks;
    }

    @Override
    public void setTasks(List<Task> tasks) {
        this.tasks = tasks;
    }

    public Task getCurrentTask() {
        return getTasks().stream()
                .filter(task -> task.getStatus() != null && task.getStatus().isActive())
                .findFirst()
                .orElse(null);
    }

    public boolean hasCurrentTask() {
        return getCurrentTask() != null;
    }

    public Release getRelease() {
        return release;
    }

    @Override
    public Integer getReleaseUid() {
        if (releaseUid != null) {
            return releaseUid;
        } else {
            Release rel = getRelease();
            if (null != rel) {
                return rel.getCiUid();
            } else {
                return null;
            }
        }
    }

    @Override
    public void setReleaseUid(Integer releaseUid) {
        this.releaseUid = releaseUid;
    }

    public String getDisplayPath() {
        return getId() + " (" + getTitle() + ")";
    }

    public void setRelease(Release release) {
        this.release = release;
    }

    @PublicApiMember
    public PhaseStatus getStatus() {
        return status;
    }

    public void setStatus(PhaseStatus status) {
        this.status = status;
    }

    @PublicApiMember
    public String getColor() {
        return color;
    }

    @PublicApiMember
    public void setColor(String color) {
        this.color = color;
    }

    public String getOriginId() {
        return originId;
    }

    public void setOriginId(final String originId) {
        this.originId = originId;
    }

    @VisibleForTesting
    void complete(Changes changes) {
        setStatus(PhaseStatus.COMPLETED);
        setEndDate(new Date());
        freezeVariables(true);
        changes.update(this);
        changes.addOperation(new PhaseCompleteOperation(this));
    }

    private void freezeVariables(boolean freezeEvenIfUnresolved) {
        if (getRelease() == null) return;

        Map<String, ValueWithInterpolation> variables = getRelease().getAllVariableValuesAsStringsWithInterpolationInfo();
        if (variables == null || variables.isEmpty()) return;

        setTitle(replaceAllWithInterpolation(getTitle(), filterOutBlankValues(variables), new HashSet<>(), freezeEvenIfUnresolved));
        setDescription(replaceAllWithInterpolation(getDescription(), variables, new HashSet<>(), freezeEvenIfUnresolved));
    }

    private Task findTopLevelTask(String taskId) {
        String topLevelFolderlessTaskId = Ids.getFolderlessId(Ids.getTopLevelTaskId(taskId));
        for (Task task : tasks) {
            if (topLevelFolderlessTaskId.equals(Ids.getFolderlessId(task.getId()))) {
                return task;
            }
        }
        return null; // not found
    }

    @VisibleForTesting
    Task findNextActivableTask() {
        for (Task task : tasks) {
            if (!task.isDone()) {
                return task;
            }
        }
        return null; // not found
    }

    public Changes start() {
        Changes changes = new Changes();

        this.setStatus(IN_PROGRESS);
        this.setStartDate(new Date());
        freezeVariables(false);
        changes.update(this);
        changes.addOperation(new PhaseStartOperation(this));

        tryStartingNextTask(changes);

        return changes;
    }

    public Changes startPendingTask(String targetTaskId) {
        Changes changes = new Changes();

        Task task = findTopLevelTask(targetTaskId);
        if (task != null) {
            changes.addAll(task.startPending(targetTaskId));
            checkTaskStatus(task, changes);
        }

        return changes;
    }

    public Changes startWithInput(String targetTaskId) {
        Changes changes = new Changes();

        Task task = getTask(targetTaskId);
        if (task != null) {
            changes.addAll(task.startWithInput());
            checkTaskStatus(task, changes);
        }

        return changes;
    }

    public Changes taskPreconditionValidated(String targetTaskId) {
        Changes changes = new Changes();

        Task task = findTopLevelTask(targetTaskId);
        if (task != null) {
            changes.addAll(task.startNow(targetTaskId, false));
            checkTaskStatus(task, changes);
        }

        return changes;
    }

    private void startActivableTask(Task firstTask, Changes changes) {
        try {
            Changes newChanges = firstTask.start();
            changes.addAll(newChanges);
            checkTaskStatus(firstTask, changes);
        } catch (Exception e) {
            // there was an error with the firstTask, but changes might contain extra information
            // related to the previous task that we have to process
            logger.error("Unable to start task '{}' with id:'{}'", firstTask.getTitle(), firstTask.getId(), e);
            changes.addAll(failTask(firstTask.getId(), e.getMessage(), User.SYSTEM));
        }
    }

    private void checkTaskStatus(Task firstTask, Changes changes) {
        if (firstTask.isDone()) {
            tryStartingNextTask(changes);
        } else if (firstTask.isFailed()) {
            changes.addAll(fail());
        }
    }

    private void tryStartingNextTask(Changes changes) {
        Task nextTask = findNextActivableTask();
        if (nextTask == null) {
            complete(changes);
        } else {
            startActivableTask(nextTask, changes);
        }
    }

    public Changes markTaskAsDone(String taskId, TaskStatus status) {
        Changes changes = new Changes();

        Task topLevelTask = findTopLevelTask(taskId);
        if (topLevelTask == null) {
            return changes;
        }

        changes.addAll(recoverPhaseIfNeeded());
        changes.addAll(topLevelTask.markAsDone(taskId, status));
        if (topLevelTask.isDone()) {
            tryStartingNextTask(changes);
        } else if (topLevelTask.isFailed()) {
            changes.addAll(fail());
        } else if (topLevelTask.isFailing()) {
            changes.addAll(failing());
        }

        return changes;
    }

    private Changes recoverPhaseIfNeeded() {
        Changes changes = new Changes();

        if (getStatus() == FAILED || getStatus() == FAILING) {
            setStatus(PhaseStatus.IN_PROGRESS);
            changes.update(this);
            changes.addOperation(new PhaseRetryOperation(this));
        }

        return changes;
    }

    public Changes failTask(String targetTaskId, String failReason, User user) {
        return failTask(targetTaskId, failReason, user, false);
    }

    public Changes failTask(String targetTaskId, String failReason, User user, boolean fromAbort) {
        Changes changes = new Changes();

        Task task = findTopLevelTask(targetTaskId);
        if (task == null) {
            return changes;
        }

        changes.addAll(task.fail(targetTaskId, failReason, user, fromAbort));
        if (task.isFailed()) {
            changes.addAll(fail());
        } else if (task.isFailing()) {
            changes.addAll(failing());
        }

        return changes;
    }

    public Changes retryTask(String taskId) {
        Changes changes = new Changes();
        Task task = findTopLevelTask(taskId);
        if (task == null) {
            return changes;
        }

        boolean taskWasInProgress = task.isInProgress();
        changes.addAll(task.retry(taskId));
        if (taskWasInProgress && task.isInProgress()) {
            // Restart was "internal" to a group, phase not impacted
            return changes;
        }

        changes.update(this);
        changes.addOperation(new PhaseRetryOperation(this));

        if (!taskWasInProgress && task.isInProgress()) {
            setStatus(IN_PROGRESS);
        } else {
            recoverPhaseIfNeeded();
            checkTaskStatus(task, changes);
        }
        return changes;
    }

    public Changes abort() {
        Changes changes = new Changes();

        this.setStatus(PhaseStatus.ABORTED);
        if (!hasStartDate()) {
            setStartDate(new Date());
        }
        setEndDate(new Date());

        changes.update(this);

        for (Task task : tasks) {
            changes.addAll(task.abort());
        }

        return changes;
    }

    public Changes fail() {
        Changes changes = new Changes();
        setStatus(FAILED);
        changes.update(this);
        changes.addOperation(new PhaseFailOperation(this));
        return changes;
    }

    public Changes failing() {
        Changes changes = new Changes();
        setStatus(FAILING);
        changes.update(this);
        changes.addOperation(new PhaseStartFailingOperation(this));
        return changes;
    }

    public Task getTask(Integer index) {
        return tasks.get(index);
    }

    public Changes resetToPlanned() {
        Changes changes = new Changes();

        setStatus(PLANNED);
        setStartDate(null);
        setEndDate(null);
        setOverdueNotified(false);

        changes.update(this);

        for (Task task : tasks) {
            changes.addAll(task.resetToPlanned());
        }

        return changes;
    }

    public boolean hasBeenStarted() {
        return status != null && status.hasBeenStarted();
    }

    public boolean isActive() {
        return status != null && status.isActive();
    }

    public boolean isDone() {
        return status == COMPLETED || status == SKIPPED;
    }

    public boolean isDefunct() {
        return isAborted() || isDone();
    }

    public boolean isUpdatable() {
        return !isDefunct();
    }

    public boolean isAborted() {
        return status == ABORTED;
    }

    public boolean isPlanned() {
        return status == PLANNED;
    }

    public boolean isFailed() {
        return status == FAILED;
    }

    public boolean isFailing() {
        return status == FAILING;
    }

    public String getReleaseOwner() {
        return release.getOwner();
    }

    public List<GateTask> getAllGates() {
        return getAllTasks().stream()
                .filter(task -> task instanceof GateTask)
                .map(task -> (GateTask) task)
                .collect(toList());
    }

    public List<Task> getAllTasks() {
        List<Task> allTasks = new ArrayList<>();

        if (tasks != null) {
            for (Task task : tasks) {
                allTasks.addAll(task.getAllTasks());
            }
        }

        return allTasks;
    }

    public List<PlanItem> getChildren() {
        List<PlanItem> children = new ArrayList<>();
        children.addAll(getAllTasks());
        return children;
    }

    public void accept(ReleaseVisitor visitor) {
        visitor.visit(this);
        for (Task task : tasks) {
            task.accept(visitor);
        }
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        return asList(
                new PropertyUsagePoint(this, "title"),
                new PropertyUsagePoint(this, "description")
        );
    }

    public Changes close() {
        Changes changes = new Changes();

        setStatus(PhaseStatus.SKIPPED);
        setStartAndEndDatesIfEmpty();

        changes.update(this);

        final Task currentTask = getCurrentTask();
        getAllTasks().stream()
                .filter(task -> !task.isDone())
                .forEach(task -> {

                    if (currentTask instanceof BaseScriptTask && currentTask.getId().equals(task.getId())) {
                        if (currentTask instanceof CustomScriptTask && ((CustomScriptTask) currentTask).isUnknown()) {
                            task.setStatus(TaskStatus.SKIPPED);
                        } else {
                            if (task.getStatus().equals(TaskStatus.IN_PROGRESS)) {
                                task.setStatus(TaskStatus.COMPLETED);
                            } else {
                                task.setStatus(TaskStatus.SKIPPED);
                            }
                        }
                    } else {
                        task.setStatus(TaskStatus.SKIPPED);
                    }

                    task.setStartAndEndDatesIfEmpty();
                    task.freezeVariables(changes, true);
                    changes.update(task);
                });

        changes.addOperation(new PhaseCloseOperation(this));
        changes.addOperation(new ReleaseAbortScriptsExecution(getRelease()));
        return changes;
    }

    public Task getTask(final String taskId) {
        return getAllTasks().stream()
                .filter(task -> Objects.equals(task.getId(), taskId))
                .findAny()
                .orElse(null);
    }

    @Override
    public void addTask(final Task task, final int position) {
        getTasks().add(position, task);
        task.setContainer(this);
    }

    public void deleteTask(Task task) {
        if (!tasks.remove(task)) {
            tasks.forEach(t -> t.deleteTask(task));
        }
    }

    public boolean isOriginal() {
        return !StringUtils.hasText(originId);
    }

    /**
     * Checks within the phase release if there is another phase that has its originId filled and referring to this phase.
     *
     * @return
     */
    public boolean isPhaseCopied() {
        List<Phase> phases = release.getPhases();
        if (phases == null) return false;

        for (Phase releasePhase : phases) {
            if (releasePhase.equals(this)) continue; // skip self
            if (getId().equals(releasePhase.getOriginId())) {
                // found some other phase with its originId pointing to this phase
                return true;
            }
        }
        return false;
    }

    public String getAncestorId() {
        if (StringUtils.hasText(getOriginId())) {
            return release.getPhase(getOriginId()).getAncestorId();
        }
        return getId();
    }

    public boolean isLatestCopy() {
        List<Phase> phases = release.getPhases();
        if (phases == null) return false;

        if (StringUtils.isEmpty(originId)) {
            // if this phase is not a copy of something (ie the very first phase), we check if we have been copied. If that is the case
            // we are not the latest.
            return !isPhaseCopied();
        }

        String ourAncestor = getAncestorId();
        List<Phase> copiesOfTheAncestorIHaveBeenCopiedFrom = new ArrayList<>();

        // we assume the order of phases is from left to right. Index 0 is the first (left), and the last phase is utmost right.
        for (Phase releasePhase : phases) {
            if (ourAncestor.equals(releasePhase.getAncestorId())) {
                // found another phase that is copied from the same ancestor as I am
                copiesOfTheAncestorIHaveBeenCopiedFrom.add(releasePhase);
            }
        }

        if (copiesOfTheAncestorIHaveBeenCopiedFrom.isEmpty()) return true;

        // the last phase we find that has the same originId, if that is us, we are the latest copy of the phase
        Phase lastPhase = copiesOfTheAncestorIHaveBeenCopiedFrom.get(copiesOfTheAncestorIHaveBeenCopiedFrom.size() - 1);
        return lastPhase == this;
    }

}
