package com.xebialabs.xlrelease.domain.runner

import com.xebialabs.deployit.plugin.api.udm.Metadata
import com.xebialabs.deployit.plugin.api.validation.{Contain, NotContain, NotEmpty, Range, Size}
import com.xebialabs.xlrelease.domain.RemoteExecution
import com.xebialabs.xlrelease.utils.XLTypes.XLProperty

import java.util
import java.util.Date
import scala.beans.{BeanProperty, BooleanBeanProperty}
import scala.jdk.CollectionConverters._

@Metadata(label = "Digital.ai Release runner", description = "Runs Release jobs in a remote environment")
class RemoteJobRunner extends JobRunner {
  @BeanProperty
  @XLProperty(required = true, label = "Runner name", password = false, description = "Runner name", readonly = true)
  @NotEmpty(message = "Runner name is required")
  @Size(min = 1, max = 255, message = "Runner name must be 255 characters or less")
  var runnerName: String = _

  @BeanProperty
  @XLProperty(required = true, password = false, description = "Runner version", readonly = true)
  @NotEmpty(message = "Runner version is required")
  @Size(min = 1, max = 255, message = "Runner version must be 255 characters or less")
  var version: String = _

  @BooleanBeanProperty
  @XLProperty(defaultValue = "true")
  var enabled: Boolean = true

  @BeanProperty
  @XLProperty(asContainment = true) // nested collections are marked 'asContainment'
  @Contain(message = "Remote runner requires default 'remote' capability", value = Array("remote"))
  @NotContain(message = "Remote runner cannot have 'local' capability", value = Array("local"))
  @Size(max = 25, message = "Remote runner cannot have more than 25 capabilities")
  var capabilities: util.Set[String] = new util.HashSet[String]()

  @BeanProperty
  @XLProperty(defaultValue = "4", description = "Capacity is number of parallel jobs")
  @Range(minimum = 1, message = "Runner capacity must be greater than 0.")
  var capacity: Integer = _

  @BeanProperty
  @XLProperty(defaultValue = "60", label = "Eviction time", description = "Number of seconds after which job container will be evicted")
  var evictionTime: Integer = _

  @BooleanBeanProperty
  @XLProperty(defaultValue = "true", label = "Encrypt job data", description = "Control job data encryption between release and remote runner")
  var encryptJobData: Boolean = true

  @BeanProperty
  @XLProperty(required = false, label = "Public key", password = true, description = "Runner public key")
  var publicKey: String = _

  @BeanProperty
  @XLProperty(required = true, label = "Idle time to live", defaultValue = "300", description = "Idle TTL is the amount of seconds a runner must confirm its aliveness. If it fails, it gets marked as unavailable. A value of 0 or less keeps it available indefinitely.")
  var idleTimeToLive: Integer = _

  @BooleanBeanProperty
  @XLProperty(defaultValue = "true")
  var available: Boolean = true

  @BeanProperty
  @XLProperty(required = false, description = "Last time when remote runner executed a job")
  var lastActivity: Date = _

  @BeanProperty
  @XLProperty(required = false, description = "PAT expiry time, updated by remote runner, do not change this field")
  var tokenExpiryDate: Date = _

  override def start(): Unit = {
    JobRunnerOps.start(this)
  }

  override def stop(): Unit = {
    JobRunnerOps.stop(this)
  }

  override def delete(): Unit = {
    JobRunnerOps.delete(this)
  }

  override def abortJob(jobId: Long): Unit = {
    JobRunnerOps.abortJob(this, jobId)
  }

  override def toString(): String = {
    val runnerCapabilities = getCapabilities().asScala.mkString(",")
    s"${this.getType.toString} [id=$getId, runnerName=$runnerName, version=$version, enabled=$enabled, capacity=$capacity, capabilities=[$runnerCapabilities]]"
  }

  override def execute(remoteExecution: RemoteExecution): String = {
    JobRunnerOps.execute(this, remoteExecution)
  }
}

object RemoteJobRunner {
  val DEFAULT_REMOTE_JOB_RUNNER_CAPABILITY: String = "remote"
}
