package com.xebialabs.xlrelease.domain;

import java.util.*;

import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.xlrelease.domain.FailureReasons.GATE_TASK_DEPENDS_ON_AN_ABORTED_RELEASE;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED_IN_ADVANCE;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.IN_PROGRESS;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.PLANNED;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;
import static java.util.stream.Collectors.toList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Gate", versioned = false, description = "Checks conditions that must be fulfilled before the release can continue")
public class GateTask extends Task {
    private static final Logger logger = LoggerFactory.getLogger(GateTask.class);

    @Property(asContainment = true, required = false)
    private List<GateCondition> conditions = new ArrayList<>();

    @Property(asContainment = true, required = false)
    private List<Dependency> dependencies = new ArrayList<>();

    @PublicApiMember
    public List<GateCondition> getConditions() {
        return conditions;
    }

    @PublicApiMember
    public void addCondition(GateCondition condition) {
        this.conditions.add(condition);
    }

    @PublicApiMember
    public GateCondition getCondition(String id) {
        return this.conditions.stream().filter(gc -> id.equals(gc.getId())).findFirst().orElse(null);
    }

    @PublicApiMember
    public boolean hasConditions() {
        return !conditions.isEmpty();
    }

    @PublicApiMember
    public void setConditions(List<GateCondition> conditions) {
        this.conditions = conditions;
    }

    @PublicApiMember
    public List<Dependency> getDependencies() {
        return this.dependencies;
    }

    @PublicApiMember
    public void setDependencies(List<Dependency> dependencies) {
        this.dependencies = new ArrayList<>(dependencies);
    }

    @PublicApiMember
    public boolean hasDependencies() {
        return !dependencies.isEmpty();
    }

    @PublicApiMember
    public void addDependency(Dependency dependency) {
        dependencies.add(dependency);
        dependency.setGateTask(this);
    }

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        Changes changes = super.execute(targetId, operation);

        if (getStatus() != IN_PROGRESS) {
            return changes;
        }

        changes.addPostAction(new ExecuteTaskAction(this));

        if (isCompletable()) {
            changes.addAll(markAsDone(getId(), COMPLETED));
        } else if (hasAbortedDependencies()) {
            String titles = getAbortedDependencyTitles();
            changes.addAll(fail(getId(), GATE_TASK_DEPENDS_ON_AN_ABORTED_RELEASE.format(titles)));
        }
        return changes;
    }

    @Override
    public Changes markAsDone(String targetId, TaskStatus targetStatus) {
        if (targetStatus == COMPLETED_IN_ADVANCE) {
            checkState(isOpenInAdvance(), "All conditions and dependencies on gate '%s' must be fulfilled before completing it.", getTitle());
        } else if (targetStatus == COMPLETED && getStatus() == COMPLETED_IN_ADVANCE) {
            checkState(isOpenWithStatus(COMPLETED_IN_ADVANCE), "All conditions and dependencies on gate '%s' must be fulfilled before completing it.", getTitle());
        } else {
            checkArgument(targetStatus != COMPLETED || isOpen(), "All conditions and dependencies on gate '%s' must be fulfilled before completing it.", getTitle());
        }
        return super.markAsDone(targetId, targetStatus);
    }

    @PublicApiMember
    public boolean isOpen() {
        return isOpenWithStatus(IN_PROGRESS);
    }

    private boolean isOpenInAdvance() {
        return isOpenWithStatus(PLANNED);
    }

    private boolean isOpenWithStatus(TaskStatus requiredStatus) {
        if (getStatus() != requiredStatus) {
            logger.trace("isOpenWithStatus for {} returns false because status {} is not equal to required {}", getId(), getStatus(), requiredStatus);
            return false;
        }

        for (GateCondition condition : conditions) {
            if (!condition.isChecked()) {
                logger.trace("isOpenWithStatus for {} returns false because condition {} is not checked", getId(), condition.getTitle());
                return false;
            }
        }

        for (Dependency dependency : dependencies) {
            if (!dependency.isDone()) {
                logger.trace("isOpenWithStatus for {} returns false because dependency {} is not done", getId(), dependency.getId());
                return false;
            }
        }

        return true;
    }

    public boolean isCompletable() {
        return (hasConditions() || hasDependencies()) && isOpen();
    }

    @Override
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables, Map<String, String> passwordVariables, Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = new HashSet<>();

        for (GateCondition condition : conditions) {
            String oldTitle = condition.getTitle();
            String newTitle = replaceAllWithInterpolation(oldTitle, variables, unresolvedVariables, freezeEvenIfUnresolved);
            if (!newTitle.equals(oldTitle)) {
                condition.setTitle(newTitle);
                changes.update(condition);
            }
        }

        dependencies.stream().filter(d -> !Strings.isNullOrEmpty(d.getTargetId()))
                .forEach(d -> {
                    String targetId = d.getTargetId();
                    String resolvedTargetId = replaceAllWithInterpolation(targetId, variables, unresolvedVariables, freezeEvenIfUnresolved);
                    if (!targetId.equals(resolvedTargetId)) {
                        logger.trace("setting targetId (but not target) to {} for GateTask {}", resolvedTargetId, getId());
                        d.setTargetId(resolvedTargetId);
                        changes.update(d);
                    }
                });

        return unresolvedVariables;
    }

    public boolean hasAbortedDependencies() {
        for (Dependency dependency : dependencies) {
            if (dependency.isAborted()) {
                return true;
            }
        }
        return false;
    }

    public String getAbortedDependencyTitles() {
        List<String> titles = new ArrayList<>();
        for (Dependency dependency : dependencies) {
            if (dependency.isAborted()) {
                titles.add(nullToEmpty(dependency.getTargetTitle()));
            }
        }

        return Joiner.on("', '").join(titles);
    }

    public void updateCondition(GateCondition gateCondition) {
        Optional<GateCondition> maybeGateCondition
                = conditions.stream().filter(condition -> condition.getId().equals(gateCondition.getId())).findFirst();
        if (maybeGateCondition.isPresent()) {
            int index = conditions.indexOf(maybeGateCondition.get());
            conditions.set(index, gateCondition);
        }

    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        usagePoints.addAll(getConditions().stream().map(condition -> new PropertyUsagePoint(condition, "title")).collect(toList()));
        usagePoints.addAll(getDependencies().stream().map(dependency -> new PropertyUsagePoint(dependency, "targetId")).collect(toList()));
        return usagePoints;
    }
}
