package com.xebialabs.xlrelease.domain.delivery.conditions;

import java.util.Date;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.delivery.Condition;
import com.xebialabs.xlrelease.domain.delivery.Delivery;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT;
import static org.joda.time.format.DateTimeFormat.mediumDateTime;

@Metadata(versioned = false, description = "Condition which completes when the specified time is reached.")
@PublicApiRef
@ShowOnlyPublicApiMembers
public class TimeCondition extends Condition {
    @Property(category = CATEGORY_INPUT, description = "Offset in hours from the delivery start date.")
    private Integer startDateOffset = 24;

    @Property(category = CATEGORY_INPUT, description = "Fixed time to transition.")
    private Date fixedDate;

    // --

    public TimeCondition() {
    }

    public TimeCondition(final Date fixedDate) {
        this.fixedDate = fixedDate;
    }


    public TimeCondition(final Integer startDateOffset) {
        this.startDateOffset = startDateOffset;
    }

    public TimeCondition(final Date fixedDate, final Integer startDateOffset) {
        this.fixedDate = fixedDate;
        this.startDateOffset = startDateOffset;
    }

    @Override
    public String getDescription() {
        if (hasFixedDate()) {
            return "Wait for fixed date: " + mediumDateTime().print(fixedDate.getTime());
        } else if (startDateOffset != null) {
            return String.format("Wait %s hours from delivery start date", startDateOffset);
        } else {
            return "Wait for an undefined date";
        }
    }

    @Override
    public void validate(Delivery delivery) {
        if (delivery.isTemplate()) {
            checkArgument(startDateOffset != null, "Offset value must be provided");
            checkArgument(startDateOffset > 0, "Offset value must be larger then 0");
        } else {
            checkArgument(hasFixedDate(), "Fixed date value must be provided");
        }
    }

    private boolean hasFixedDate() {
        return fixedDate != null;
    }

    // --

    public Date getFixedDate() {
        return fixedDate;
    }

    public void setFixedDate(final Date fixedDate) {
        this.fixedDate = fixedDate;
    }

    public Integer getStartDateOffset() {
        return startDateOffset;
    }

    public void setStartDateOffset(final Integer startDateOffset) {
        this.startDateOffset = startDateOffset;
    }
}

