package com.xebialabs.xlrelease.domain.events

import com.xebialabs.xlrelease.domain.{Task, TaskGroup}

import java.util.{List => JList}

/**
  * Common interface for domain events related to task execution or modification.
  */
sealed trait TaskEvent extends XLReleaseEvent

//
// Task CUD
//

/**
  * Event published when a task is created.
  *
  * @param task the task that was created.
  */
case class TaskCreatedEvent(override val task: Task) extends TaskCreatedOrTypeChangedEvent(task)

/**
  * Event published when a task is copied.
  *
  * @param taskCopied the created copy of the task.
  */
case class TaskCopiedEvent(taskCopied: Task) extends TaskCreatedOrTypeChangedEvent(taskCopied)

/**
 * Event published when a task is copied from another container.
 *
 * @param taskCopied the created copy of the task.
 * @param sourceContainerType the type of container that the task was copied from.
 * @param sourceContainerTitle the title of the container that the task was copied from.
 * @param sourceTaskTile the title of the source task that was copied.
 */
case class TaskCopiedFromEvent(taskCopied: Task, sourceContainerType: String, sourceContainerTitle: String) extends TaskCreatedOrTypeChangedEvent(taskCopied)

/**
  * Event published when a task is moved withing a release. It can be moved within the same container
  * (phase or a task group), or between containers (for example, to another phase).
  *
  * @param movedTask         the task that has been moved.
  * @param originIndex       the original index of the task in the original container, 0-based.
  * @param targetIndex       the new index of the task in the new container, 0-based.
  * @param originalTaskId    the original task ID. It will be same if task is moved within container.
  * @param originContainerId the original container ID of the task: a phase ID or a task group ID.
  * @param targetContainerId the new container ID of the task.
  */
case class TaskMovedEvent(movedTask: Task, originIndex: Int, targetIndex: Int, originalTaskId: String,
                          originContainerId: String, targetContainerId: String) extends TaskEvent

/**
  * Event published when a task's properties have been updated.
  *
  * @param original a copy of the task before it was updated.
  * @param updated  the updated copy of the task. You can use [[original]] to compare which properties have changed.
  */
case class TaskUpdatedEvent(original: Task, updated: Task) extends TaskCreatedOrTypeChangedEvent(updated)

/**
  * Event published when a task has been deleted.
  *
  * @param task a copy of the deleted task.
  */
case class TaskDeletedEvent(task: Task) extends TaskEvent

/**
 * Event published when the task status line has been updated. It could be that it's the same message as before.
 * @param taskId identifier of the task.
 * @param statusLine the status line of the task.
 */
case class TaskStatusLineUpdated(taskId: String, statusLine: String) extends TaskEvent

//
// Task execution
//

/**
  * Common interface for events when the task status changes.
  */
sealed trait TaskExecutionEvent extends TaskEvent {
  /**
    * The task of which status has changed.
    */
  val task: Task
}

/**
  * Event published when a task has started.
  *
  * @param task instance of the task.
  */
case class TaskStartedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published when a task goes into `WAITING_FOR_INPUT` ("needs input") state. It happens when
  * a task uses some required variables for which no value has been supplied yet in the release.
  *
  * @param task             instance of the task.
  * @param unboundVariables list of variable names for which require a non-empty value before this task can be started.
  */
case class TaskWaitingForInputEvent(task: Task, unboundVariables: JList[String]) extends TaskExecutionEvent

/**
  * Event published when a task has been completed.
  *
  * @param task      instance of the task.
  * @param inAdvance `true` if this task was completed in advance, `false` if it has just finished executing.
  */
case class TaskCompletedEvent(task: Task, inAdvance: Boolean) extends TaskExecutionEvent

/**
  * Event published when a task has been skipped.
  *
  * @param task      instance of the task.
  * @param inAdvance `true` if this task was skipped in advance, `false` if it was `IN_PROGRESS` and then skipped.
  */
case class TaskSkippedEvent(task: Task, inAdvance: Boolean) extends TaskExecutionEvent

/**
  * Event published when a task has been aborted. This happens when a release is aborted: then all its unfinished
  * tasks are also aborted.
  *
  * @param task instance of the task.
  */
case class TaskAbortedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published when a task has been retried. For a task group it happens when the group was failing or failed and
  * one of its task is retried. For other tasks it means that the task was retried after a failure.
  *
  * @param task instance of the task.
  */
case class TaskRetriedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published when a task enters `PENDING` state, which means that it has a scheduled start date in the future.
  *
  * @param task instance of the task.
  */
case class TaskDelayedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published when a task has failed.
  *
  * @param task   instance of the task.
  * @param reason the optional reason of the failure: either provided by a user or by the system.
  */
case class TaskFailedEvent(task: Task, reason: String) extends TaskExecutionEvent

/**
  * Event published when a task is reopened: it was completed or skipped in advance and goes back into planned state.
  *
  * @param task instance of the task.
  */
case class TaskReopenedEvent(task: Task) extends TaskExecutionEvent


/**
  * Event published after a task failed and recovery started.
  *
  * @param task instance of the task.
  */
case class TaskRecoveryStartedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published after a task finished recovering.
  *
  * @param task instance of the task.
  */
case class TaskRecoveredEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published after a task abort script started.
  *
  * @param task instance of the task.
  */
case class TaskAbortScriptStartedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published after a task abort script has completed.
  *
  * @param task instance of the task.
  */
case class TaskAbortScriptCompletedEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published when a parallel or sequential group goes into the "failing" state: so it has some tasks which are failed and
  * some tasks which are still running.
  *
  * @param task instance of the task.
  */
case class TaskGroupFailingEvent(task: TaskGroup) extends TaskExecutionEvent

/**
  * Event published when a task is overdue.
  *
  * @param task instance of the task.
  */
case class TaskOverdueEvent(task: Task) extends TaskExecutionEvent

/**
  * Event published when a task is due soon.
  *
  * @param task instance of the task.
  */
case class TaskDueSoonEvent(task: Task) extends TaskExecutionEvent


/**
  * Event published when tasks are locked.
  *
  * @param releaseId id of a release that contains locked tasks.
  * @param tasks     locked tasks.
  */
case class TasksLockedEvent(releaseId: String, tasks: JList[Task]) extends TaskEvent

/**
  * Event published when tasks are unlocked.
  *
  * @param releaseId id of a release that contains locked tasks.
  * @param tasks     unlocked tasks.
  */
case class TasksUnlockedEvent(releaseId: String, tasks: JList[Task]) extends TaskEvent

abstract class TaskCreatedOrTypeChangedEvent(val task: Task) extends TaskEvent


/**
 * Event published when a task job execution is done regardless success or failure
 *
 * @param taskId      the taskId
 * @param executionId the executionId that is done.
 */
case class TaskJobExecutedEvent(taskId: String, executionId: String) extends TaskEvent
