package com.xebialabs.xlrelease.domain;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.xlrelease.domain.PythonScriptDefinition.ofKind;

/**
 * Base class for the configuration of a custom Python task.
 * <p>
 * Users subclass this in <code>synthetic.xml</code> to configure input parameters and output mappings for their custom script tasks.
 * </p>
 * <p>
 * Note: this should be called "PythonScriptDefinition", but was renamed for UX reasons since the type name is exposed to end users.
 * </p>
 */

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(versioned = false)
public class PythonScript extends BasePythonScript {

    @Property(required = false, asContainment = true)
    private CustomScriptTask customScriptTask;

    @Property(hidden = true, required = false, description = "Link with documentation about this task.")
    private String documentationPage;

    @Property(hidden = true, defaultValue = "32768", description = "Maximum size of output property values of type string. Default value is 32768")
    private int maxOutputPropertySize = 32768;

    @Property(hidden = true, required = false, defaultValue = "true", category = "internal", description = "The task is supported in a workflow or not")
    private boolean supportedInWorkflow;

    @Property(hidden = true, isTransient = true, order = Integer.MAX_VALUE, category = "internal", label = "Script timeout", description = "Maximum execution time (in seconds) allowed for each individual script", defaultValue = "-1")
    private int scriptTimeout;

    public static final String PYTHON_SCRIPT_ID = "PythonScript";
    public static final String PYTHON_SCRIPT_PROPERTY = "pythonScript";
    public static final String CUSTOM_SCRIPT_TASK_PROPERTY = "customScriptTask";

    private static final String TASK_COLOR_PROPERTY = "taskColor";
    private static final String ICON_CLASS_PROPERTY = "iconClass";
    private static final String ICON_LOCATION_PROPERTY = "iconLocation";
    private static final String OUTPUT_VAR_HELP_TEXT_PROPERTY = "outputVarHelpText";
    private static final String SUPPORTED_IN_WORKFLOW_PROPERTY = "supportedInWorkflow";

    public static final List<String> UPDATEABLE_PROPERTIES = Arrays.asList(TASK_COLOR_PROPERTY,
            ICON_CLASS_PROPERTY, ICON_LOCATION_PROPERTY, OUTPUT_VAR_HELP_TEXT_PROPERTY, SUPPORTED_IN_WORKFLOW_PROPERTY);

    public Collection<PropertyDescriptor> getPropertiesWithVariables() {
        PythonScriptDefinition scriptDefinition = new PythonScriptDefinition(getType());
        return scriptDefinition.getInputProperties()
                .stream()
                .filter(ofKind(STRING, LIST_OF_STRING, SET_OF_STRING, MAP_STRING_STRING))
                .collect(Collectors.toList());
    }

    public Collection<PropertyDescriptor> getInputProperties() {
        return new PythonScriptDefinition(getType(), true).getInputProperties();
    }

    public Collection<PropertyDescriptor> getOutputProperties() {
        return new PythonScriptDefinition(getType(), true).getOutputProperties();
    }

    public Collection<PropertyDescriptor> getTransitionalAndOutputProperties() {
        return new PythonScriptDefinition(getType(), true).getTransitionalAndOutputProperties();
    }

    public Collection<PropertyDescriptor> getTransitionalProperties() {
        return new PythonScriptDefinition(getType(), true).getTransitionalProperties();
    }

    @PublicApiMember
    public CustomScriptTask getCustomScriptTask() {
        return customScriptTask;
    }

    @PublicApiMember
    public void setCustomScriptTask(CustomScriptTask customScriptTask) {
        this.customScriptTask = customScriptTask;
    }

    @PublicApiMember
    public String getIconLocation() {
        return getProperty(ICON_LOCATION_PROPERTY);
    }

    @PublicApiMember
    public String getOutputVarHelpText() {
        return getProperty("outputVarHelpText");
    }

    @PublicApiMember
    public String getTaskColor() {
        return getProperty(TASK_COLOR_PROPERTY);
    }

    @PublicApiMember
    public String getIconClass() {
        return getProperty(ICON_CLASS_PROPERTY);
    }

    public int getMaxOutputPropertySize() {
        return maxOutputPropertySize;
    }

    public void setMaxOutputPropertySize(final int maxOutputPropertySize) {
        this.maxOutputPropertySize = maxOutputPropertySize;
    }

    public String getDocumentationPage() {
        return documentationPage;
    }

    public void setDocumentationPage(String documentationPage) {
        this.documentationPage = documentationPage;
    }

    public boolean isSupportedInWorkflow() {
        return supportedInWorkflow;
    }

    public void setSupportedInWorkflow(final boolean supportedInWorkflow) {
        this.supportedInWorkflow = supportedInWorkflow;
    }

    @Override
    public String getScript() throws IOException {
        return ScriptHelper.readScript(customScriptTask.getScriptPath());
    }

    @Override
    public String getAbortScript() throws IOException {
        return ScriptHelper.readScript(customScriptTask.getAbortScriptPath());
    }

    public int getScriptTimeout() {
        return scriptTimeout;
    }

    public void setScriptTimeout(int scriptTimeout) {
        this.scriptTimeout = scriptTimeout;
    }
}
