package com.xebialabs.xlrelease.domain;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.events.TaskFailOperation;
import com.xebialabs.xlrelease.events.TaskGroupFailingOperation;
import com.xebialabs.xlrelease.events.TaskRetryOperation;
import com.xebialabs.xlrelease.events.XLReleaseOperation;
import com.xebialabs.xlrelease.user.User;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.*;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Task Group (Parallel or Sequential)", versioned = false, virtual = true)
public abstract class TaskGroup extends Task implements TaskContainer {

    @Property(asContainment = true, required = false)
    protected List<Task> tasks = newArrayList();

    @PublicApiMember
    public List<Task> getTasks() {
        return tasks;
    }

    @PublicApiMember
    public abstract void setTasks(List<Task> tasks);

    @PublicApiMember
    public Task getTask(Integer index) {
        return tasks.get(index);
    }

    @PublicApiMember
    @Override
    public List<Task> getAllTasks() {
        List<Task> allTasks = new ArrayList<>(super.getAllTasks());
        if (tasks != null) {
            for (Task task : tasks) {
                allTasks.addAll(task.getAllTasks());
            }
        }
        return allTasks;
    }

    @Override
    public List<PlanItem> getChildren() {
        List<PlanItem> children = new ArrayList<>();
        if (tasks != null) {
            for (Task task : tasks) {
                children.addAll(task.getAllTasks());
            }
        }
        return children;
    }

    @Override
    public void accept(ReleaseVisitor visitor) {
        super.accept(visitor);
        for (Task task : tasks) {
            task.accept(visitor);
        }
    }

    private boolean containsTaskId(final String targetId, final Task subTask) {
        return subTask.getAllTasks().stream().anyMatch(task -> task.getId().equals(targetId));
    }

    protected abstract Changes tryToStartPlanningTargets(Task task);

    protected abstract Changes startSubTasksIfPreconditionNotInProgress();

    @Override
    public Changes startNow(String targetId, boolean shouldBePending) {
        Changes changes = new Changes();

        if (getId().equals(targetId)) {
            changes.addAll(super.startNow(getId(), shouldBePending));
            if (getStatus() == FAILED || getStatus() == PENDING) {
                return changes;
            }

            changes.addAll(startSubTasksIfPreconditionNotInProgress());
        } else {
            for (Task subTask : tasks) {
                if (containsTaskId(targetId, subTask)) {
                    changes.addAll(subTask.startNow(targetId, shouldBePending));
                    changes.addAll(tryToStartPlanningTargets(subTask));
                    break;
                }
            }
        }
        updateGroupStatusIfNecessary(changes);
        return changes;
    }

    @Override
    public Changes markAsDone(String targetId, TaskStatus status) {
        Changes changes = new Changes();

        if (getId().equals(targetId)) {
            changes.addAll(super.markAsDone(targetId, status));
            for (Task subTask : tasks) {
                changes.addAll(subTask.markAsDone(subTask.getId(), status));
            }
        } else {
            for (Task subTask : tasks) {
                if (containsTaskId(targetId, subTask)) {
                    changes.addAll(subTask.markAsDone(targetId, status));
                    changes.addAll(tryToStartPlanningTargets(subTask));
                    break;
                }
            }
            updateGroupStatusIfNecessary(changes);
        }

        return changes;
    }

    @Override
    public Changes markAsFailing(final String targetId, final String failingReason) {
        Changes changes = new Changes();
        if (getId().equals(targetId)) {
            changes.addAll(super.markAsFailing(targetId, failingReason));
        } else {
            for (Task subTask : tasks) {
                if (containsTaskId(targetId, subTask)) {
                    changes.addAll(subTask.markAsFailing(targetId, failingReason));
                    break;
                }
            }
            updateGroupStatusIfNecessary(changes);
        }

        return changes;
    }

    @Override
    public Changes fail(String targetId, String failReason) {
        return fail(targetId, failReason, User.AUTHENTICATED_USER, false);
    }

    @Override
    public Changes fail(String targetId, String failReason, boolean fromAbort) {
        return fail(targetId, failReason, User.AUTHENTICATED_USER, fromAbort);
    }

    @Override
    public Changes fail(String targetId, String failReason, User user, boolean fromAbort) {
        Changes changes = new Changes();
        if (getId().equals(targetId)) {
            changes.addAll(super.fail(targetId, failReason, user, fromAbort));
            setFailuresCount(getFailuresCount() + 1);
        } else {
            for (Task subTask : tasks) {
                if (containsTaskId(targetId, subTask)) {
                    changes.addAll(subTask.fail(targetId, failReason, user, fromAbort));
                    break;
                }
            }
            updateGroupStatusIfNecessary(changes);
        }

        return changes;
    }

    @Override
    public Changes abort() {
        Changes changes = super.abort();
        for (Task subTask : tasks) {
            changes.addAll(subTask.abort());
        }
        return changes;
    }

    @Override
    public Changes resetToPlanned() {
        Changes changes = super.resetToPlanned();
        for (Task task : getTasks()) {
            changes.addAll(task.resetToPlanned());
        }
        return changes;
    }

    @Override
    public Changes retry(String targetId) {
        Changes changes = new Changes();

        if (getId().equals(targetId)) {
            changes.addAll(super.retry(targetId));

            if (getStatus() != FAILED && getStatus() != PENDING) {
                changes.addAll(startSubTasksIfPreconditionNotInProgress());
                updateGroupStatusIfNecessary(changes);
            }

        } else {
            for (Task subTask : tasks) {
                if (containsTaskId(targetId, subTask)) {
                    changes = subTask.retry(targetId);
                    changes.addAll(tryToStartPlanningTargets(subTask));
                    break;
                }
            }

            updateGroupStatusIfNecessary(changes);
        }

        return changes;
    }

    protected void updateGroupStatusIfNecessary(Changes changes) {
        if (isPreconditionInProgress() || isFacetInProgress()) {
            return;
        }

        TaskStatus newGroupStatus = computeGroupStatus(tasks);
        if (newGroupStatus != this.getStatus()) {
            setStatus(newGroupStatus);
            if (newGroupStatus == IN_PROGRESS && !hasStartDate()) {
                setStartDate(new Date());
            }
            if (newGroupStatus == COMPLETED) {
                setEndDate(new Date());
                changes.addAll(super.markAsDone(this.getId(), getStatus()));
            } else {
                if (newGroupStatus == FAILED) {
                    setFailuresCount(getFailuresCount() + 1);
                }
                changes.addOperation(getOperationForNewStatus(newGroupStatus));
            }
            changes.update(this);
        }
    }

    @VisibleForTesting
    static TaskStatus computeGroupStatus(List<Task> tasks) {
        boolean hasFailures = false;
        boolean hasInProgress = false;
        for (Task subTask : tasks) {
            if (subTask.isPending() ||
                subTask.isWaitingForInput() ||
                subTask.isPreconditionInProgress() ||
                subTask.isInProgress() ||
                subTask.isFailing() ||
                subTask.isFailureHandlerInProgress() ||
                subTask.isFacetInProgress()) {
                hasInProgress = true;
            }
            if (subTask.isFailed() || subTask.isFailing()) {
                hasFailures = true;
            }
        }

        if (hasFailures && hasInProgress) {
            return FAILING;
        } else if (hasFailures) {
            return FAILED;
        } else if (hasInProgress) {
            return IN_PROGRESS;
        } else {
            return COMPLETED;
        }
    }

    private XLReleaseOperation getOperationForNewStatus(TaskStatus status) {
        switch (status) {
            case FAILED:
                return new TaskFailOperation(this, "at least one subtask failed");
            case IN_PROGRESS:
                return new TaskRetryOperation(this);
            case FAILING:
                return new TaskGroupFailingOperation(this);
            default:
                throw new IllegalArgumentException(status.name());
        }
    }

    @Override
    public boolean isUpdatable() {
        return !isDefunct();
    }

    public boolean isSkippableOrRetriable() {
        for (PlanItem planItem : getChildren()) {
            Task task = (Task) planItem;
            if (!task.isPlanned() && !task.isDoneInAdvance()) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int getFlaggedCount() {
        int flagsCount = 0;
        for (Task subTask : getAllTasks()) {
            if (subTask.hasBeenFlagged()) {
                flagsCount++;
            }
        }
        return flagsCount;
    }

    @Override
    public int getDelayedCount() {
        int delaysCount = 0;
        for (Task subTask : getAllTasks()) {
            if (subTask.hasBeenDelayed()) {
                delaysCount++;
            }
        }
        return delaysCount;
    }

    @Override
    protected List<String> getUnboundRequiredVariables() {
        return Collections.emptyList();
    }

    @Override
    public void deleteTask(Task task) {
        if (!tasks.remove(task)) {
            tasks.forEach(t -> t.deleteTask(task));
        }
    }

    public void addTask(Task task, int position) {
        getTasks().add(position, task);
        task.setContainer(this);
    }

    @Override
    public boolean isFailureHandlerEnabled() {
        return false;
    }
}
