package com.xebialabs.xlrelease.variable

import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.utils.Graph
import com.xebialabs.xlrelease.utils.Graph.Edge
import grizzled.slf4j.Logging

import java.util.{List => JList, Map => JMap, Set => JSet}
import scala.jdk.CollectionConverters._


object VariablesGraphResolver extends Logging {

  type Ref = String

  case class Var[A, V <: Variable](value: A, variable: V, resolved: Boolean)

  // Scope map, accessing a variable, its value and its resolved status by reference.
  type Scope = Map[Ref, Var[_, _ <: Variable]]
  // Dependencies map. Used to build graph edges.
  type Deps = Map[Ref, Set[Ref]]

  // using the `variables` parameter, resolve interconnected variables.
  // resolved variables go in the first map, those that are impossible to resolve will go on the second one
  def resolveAsStrings(variables: Set[Variable]): (Map[String, String], Map[String, String]) = {
    val scope = resolve(variables)
    val (resolved, unresolved) = scope.foldLeft(Map.empty[Ref, Var[_, _ <: Variable]] -> Map.empty[Ref, Var[_, _ <: Variable]]) {
      case ((res, unres), (ref, v@Var(_, _, true))) => (res + (ref -> v)) -> unres
      case ((res, unres), (ref, v@Var(_, _, false))) => res -> (unres + (ref -> v))
    }

    toReplacement(resolved) -> toReplacement(unresolved)
  }

  def resolve(variables: Set[Variable]): Scope = {
    val scope0: Scope = scopeFor(variables)
    val deps: Deps = dependenciesFor(scope0)

    val scope1 = resolveNoDeps(deps).apply(scope0)

    resolveDeps(deps).apply(scope1)
  }

  // mark all variables with no deps as resolved
  private def resolveNoDeps(deps: Deps): Scope => Scope = scope => {
    deps.filter(_._2.isEmpty).keySet.foldLeft(scope) {
      case (s, ref) =>
        s.get(ref).fold(s) {
          case v if v.variable.isValueEmpty && v.variable.getRequiresValue => s
          case v => s.updated(ref, v.copy(resolved = true))
        }
    }
  }

  // resolve variables with dependencies
  private def resolveDeps(deps: Deps): Scope => Scope = {
    val graph = graphFor(deps)

    // state.value here is a Scope => Scope function. Initialized as identity[Scope].
    Graph.walk(graph)(identity[Scope](_)) {
      case (state, ref) if !state.hasCycle =>
        s =>
          state.value(
            // resolve this ref, the result will be used as input for the already existing state.value function
            resolveFor(ref, graph).apply(s)
          )
      case (state, ref) =>
        s =>
          state.value(
            // cycle detected, try to resolve variables as much as it is possible
            resolveRemaining(ref, graph).apply(s)
          )
    }.value // return the complete Scope => Scope function built during the walk.
  }

  // resolve the value for a single reference
  private def resolveFor(ref: Ref, graph: Graph[Ref]): Scope => Scope = {
    val deps = graph.incoming(ref)
    if (deps.isEmpty) {
      // trying to resolve a variable without dependencies. If it was not resolved yet, it cannot be done.
      identity[Scope]
    } else {
      scope: Scope => {
        scope.get(ref).fold(scope) { v =>
          // resolve is boolean denoting the fact that we did not find any unresolved variable as part of its dependencies
          val (replacements, resolve) = replacementsFor(ref)(scope, deps)
          val (s1, newValueAsString) = update(ref, v, replacements.asJava, markAsResolved = resolve).apply(scope)
          if (resolve) {
            // replace this variable's key with the newValueAsString on all references that depend on it
            val partialReplacement = Map(VariableHelper.withVariableSyntax(ref) -> newValueAsString).asJava
            graph.outgoing(ref).filterNot(_ == ref).foldLeft(s1) { case (sN, dep) =>
              sN.get(dep).fold(sN) { dv =>
                update(dep, dv, partialReplacement, markAsResolved = false).apply(sN)._1
              }
            }
          } else {
            s1
          }
        }
      }
    }
  }

  // use the resolved variables to resolve the not-yet-resolved-because-cycles ones as much as possible
  private def resolveRemaining(ref: Ref, graph: Graph[Ref]): Scope => Scope = scope => {
    def done: Scope = scope.filter(_._2.resolved)

    def todo: Deps = scope.filterNot(_._2.resolved).keySet.map {
      ref => ref -> graph.incoming(ref)
    }.toMap

    todo.filter(_._2.forall(done.keySet.contains)).keySet.foldLeft(scope) {
      case (s1, ref1) => resolveFor(ref1, graph).apply(s1)
    }
  }

  // update a variable's value using replacements
  private def update[A, V <: Variable](ref: Ref, entry: Var[A, V],
                                       replacements: JMap[String, String],
                                       markAsResolved: Boolean = true): Scope => (Scope, String) = {
    val newValue = (entry.value, entry.variable) match {
      case (value: String, _: StringVariable) =>
        VariableHelper.replaceAll(value, replacements)
      case (value: JList[String@unchecked], _: ListStringVariable) =>
        value.asScala.map(VariableHelper.replaceAll(_, replacements)).asJava
      case (value: JMap[String@unchecked, String@unchecked], _: MapStringStringVariable) =>
        value.asScala.view.mapValues(VariableHelper.replaceAll(_, replacements)).toMap.asJava
      case (value: JSet[String@unchecked], _: SetStringVariable) =>
        value.asScala.map(VariableHelper.replaceAll(_, replacements)).asJava
      case (value, _: PasswordStringVariable) =>
        value
      case (value, _: DateVariable) =>
        value
      case (value, _: IntegerVariable) =>
        value
      case (value, _: BooleanVariable) =>
        value
      case (other, v: Variable) =>
        val otherClassOrNull = Option(other).map(_.getClass.getSimpleName).getOrElse("null")
        val vClassOrNull = Option(v).map(_.getClass.getSimpleName).getOrElse("null")
        logger.warn(s"Invalid combination: $otherClassOrNull as $vClassOrNull value")
        other
    }
    s => s.updated(ref, entry.copy(value = newValue, resolved = markAsResolved)) -> valueAsString(entry.variable, newValue)
  }

  // temporarily set the variable value to the given value to return its string representation, restore the original value afterwards
  private[variable] def valueAsString[V <: Variable](variable: V, value: Any): String = {
    // make a shallow copy of variable and use that clone
    val clonedVariable = variable.clone()
    clonedVariable.setUntypedValue(value)
    val valueAsString = if (clonedVariable.isValueEmpty) {
      clonedVariable.getEmptyValueAsString
    } else {
      clonedVariable.getValueAsString
    }
    valueAsString
  }

  // given the whole scope, return a Map[String, String] using the variableSyntax for keys and the string representation for values
  // discard empty values, but only if they are not marked as 'required'.
  private def toReplacement(scope: Scope): Map[String, String] = {
    scope.values.collect {
      case Var(value, variable, _) if !variable.isValueEmpty || !variable.getRequiresValue =>
        VariableHelper.formatVariableIfNeeded(variable.getKey) -> valueAsString(variable, value)
    }.toMap
  }

  // build a Scope map from a set of variables
  private[variable] def  scopeFor(variables: Set[Variable]): Scope =
    variables.map(v => v.getKey -> Var(Option(v.getValue).getOrElse(v.getEmptyValue), v, resolved = false)).toMap

  // build the Deps map from a Scope map
  private[variable] def dependenciesFor(scope: Scope): Deps = {
    scope.map { case (ref, Var(_, variable, _)) =>
      ref -> {
        for {
          value: String <- Option(variable).filterNot(_.isValueEmpty).map(_.getValueAsString).toSet
          key: String <- VariableHelper.collectVariables(value).asScala.toSet
          depRef <- scope.keySet
          if depRef == VariableHelper.withoutVariableSyntax(key)
        } yield depRef
      }
    }
  }

  // build a Graph[Ref] for the given Deps
  private[variable] def graphFor(deps: Deps): Graph[Ref] = Graph {
    for {
      ref <- deps.keySet
      deps: Set[Ref] <- deps.get(ref).toSet
      dep <- deps
    } yield Edge(dep -> ref)
  }

  private def  replacementsFor(ref: Ref)(scope: Scope, dependencies: Set[Ref]): (Map[String, String], Boolean) = {
    val replacements: Set[(String, Option[String])] = for {
      dep <- dependencies
      r <- scope.get(dep).map { entry =>
        val value = if (entry.resolved) {
          Some(entry.variable.getValueAsString)
        } else {
          logger.warn(s"unresolved dependency $dep[${entry.variable.getValue}] for $ref")
          None
        }
        VariableHelper.withVariableSyntax(dep) -> value
      }
    } yield r

    // false if at least one unresolved dependency was found, true otherwise.
    replacements.foldLeft(Map.empty[String, String] -> true) {
      case ((m, ok), (key, maybeValue)) => maybeValue match {
        case None => m -> false
        case Some(value) => (m + (key -> value)) -> ok
      }
    }
  }

}
