package com.xebialabs.xlrelease.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.notification.MailPriority;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.IN_PROGRESS;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAll;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;
import static java.util.Arrays.asList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Notification", versioned = false, description = "Sends notification messages via email to specified recipients")
public class NotificationTask extends Task {

    @Property(description = "List of email addresses where the message will be sent.")
    private Set<String> addresses;

    @Property(description = "Subject of the message.")
    private String subject;

    @Property(description = "Message body, in Markdown format.")
    private String body;

    @Property(required = false, description = "List of email addresses that receive the message as CC.")
    private Set<String> cc;

    @Property(required = false, description = "List of email addresses that receive the message as BCC.")
    private Set<String> bcc;

    @Property(required = false, description = "The email address of the reply message recipient")
    private String replyTo;

    @Property(required = false, description = "The email address of the message sender.")
    private String from;

    @Property(required = false, description = "The SMTP server username of the message sender.")
    private String senderUsername;

    @Property(required = false, password = true, description = "The SMTP server password of the message sender.")
    private String senderPassword;

    @Property(required = false, description = "Priority of the message.")
    private MailPriority priority;

    @Override
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables,
                                                        Map<String, String> passwordVariables, @SuppressWarnings("unused") Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = newHashSet();

        setAddresses(replaceAllInSet(getAddresses(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setSubject(replaceAllWithInterpolation(getSubject(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setBody(replaceAllWithInterpolation(getBody(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setCc(replaceAllInSet(getCc(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setBcc(replaceAllInSet(getBcc(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setReplyTo(replaceAllWithInterpolation(getReplyTo(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setFrom(replaceAllWithInterpolation(getFrom(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setSenderUsername(replaceAllWithInterpolation(getSenderUsername(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setSenderPassword(replaceAll(getSenderPassword(), passwordVariables, unresolvedVariables, freezeEvenIfUnresolved));

        return unresolvedVariables;
    }

    private Set<String> replaceAllInSet(Set<String> setOfString, Map<String, ValueWithInterpolation> variables, Set<String> unresolvedVariables, boolean freezeEvenIfUnresolved) {
        Set<String> newSetOfString = newHashSet();
        for (String item : setOfString) {
            newSetOfString.add(replaceAllWithInterpolation(item, variables, unresolvedVariables, freezeEvenIfUnresolved));
        }
        return newSetOfString;
    }

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        Changes changes = super.execute(targetId, operation);

        if (getStatus() != IN_PROGRESS) {
            return changes;
        }

        generateExecutionId();
        changes.update(this);

        changes.addPostAction(new ExecuteTaskAction(this));

        return changes;
    }

    @PublicApiMember
    public Set<String> getAddresses() {
        return addresses;
    }

    @PublicApiMember
    public void setAddresses(Set<String> addresses) {
        this.addresses = addresses;
    }

    @PublicApiMember
    public String getSubject() {
        return subject;
    }

    @PublicApiMember
    public void setSubject(String subject) {
        this.subject = subject;
    }

    @PublicApiMember
    public String getBody() {
        return body;
    }

    @PublicApiMember
    public void setBody(String body) {
        this.body = body;
    }

    @PublicApiMember
    public Set<String> getCc() {
        return cc;
    }

    @PublicApiMember
    public void setCc(final Set<String> cc) {
        this.cc = cc;
    }

    @PublicApiMember
    public Set<String> getBcc() {
        return bcc;
    }

    @PublicApiMember
    public void setBcc(final Set<String> bcc) {
        this.bcc = bcc;
    }

    @PublicApiMember
    public String getReplyTo() {
        return replyTo;
    }

    @PublicApiMember
    public void setReplyTo(final String replyTo) {
        this.replyTo = replyTo;
    }

    @PublicApiMember
    public String getFrom() {
        return from;
    }

    @PublicApiMember
    public void setFrom(final String from) {
        this.from = from;
    }

    @PublicApiMember
    public String getSenderUsername() {
        return senderUsername;
    }

    @PublicApiMember
    public void setSenderUsername(String senderUsername) {
        this.senderUsername = senderUsername;
    }

    @PublicApiMember
    public String getSenderPassword() {
        return senderPassword;
    }

    @PublicApiMember
    public void setSenderPassword(String senderPassword) {
        this.senderPassword = senderPassword;
    }

    @PublicApiMember
    public MailPriority getPriority() {
        return priority;
    }

    @PublicApiMember
    public void setPriority(final MailPriority priority) {
        this.priority = priority;
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        usagePoints.addAll(asList(
            new PropertyUsagePoint(this, "addresses"),
            new PropertyUsagePoint(this, "subject"),
            new PropertyUsagePoint(this, "body"),
            new PropertyUsagePoint(this, "bcc"),
            new PropertyUsagePoint(this, "cc"),
            new PropertyUsagePoint(this, "replyTo"),
            new PropertyUsagePoint(this, "priority")
        ));
        return usagePoints;
    }
}


