package com.xebialabs.xlrelease.domain.blackout

import com.xebialabs.xlrelease.api.internal.InternalMetadata
import com.xebialabs.xlrelease.domain.blackout.BlackoutMetadata.toStartOfNextMinute
import org.joda.time.DateTime

import java.util.{Collections, Date, List => JList}
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

object BlackoutMetadata {
  def apply(periods: JList[BlackoutPeriod]): BlackoutMetadata = new BlackoutMetadata(periods)
  def apply(periods: Seq[BlackoutPeriod]): BlackoutMetadata = new BlackoutMetadata(periods.asJava)

  lazy val empty: BlackoutMetadata = new BlackoutMetadata()

  val BLACKOUT = "blackout"

  def toStartOfNextMinute(date: Date): Date = {
    new DateTime(date).plusMinutes(1).withSecondOfMinute(0).withMillisOfSecond(0).toDate
  }
}

class BlackoutMetadata(periods: JList[BlackoutPeriod] = Collections.emptyList[BlackoutPeriod]()) extends InternalMetadata {

  def isInBlackout(now: Date): Boolean = {
    periods.asScala.exists(_.contains(now))
  }

  def getEndOfBlackout(now: Date): Date = periods.asScala
    .find(_.contains(now))
    .map(unionWithOverlapping(_, periods.asScala.toSeq).endDate)
    .map(toStartOfNextMinute)
    .orNull

  private def unionWithOverlapping(period: BlackoutPeriod, periods: Seq[BlackoutPeriod]): BlackoutPeriod = {
    periods.sortBy(_.startDate).foldLeft(period)((merged, current) => merged.unionWith(current))
  }
}

object BlackoutPeriod {
  implicit def toLong(date: Date): Long = date.getTime
}

case class BlackoutPeriod(startDate: Date, endDate: Date) {

  import BlackoutPeriod._

  def contains(now: Date): Boolean = {
    startDate <= now && endDate >= now
  }

  def overlapsWith(other: BlackoutPeriod): Boolean = {
    startDate <= other.endDate && endDate >= other.startDate
  }

  def unionWith(other: BlackoutPeriod): BlackoutPeriod = {
    if (overlapsWith(other)) {
      BlackoutPeriod(Seq(startDate, other.startDate).min, Seq(endDate, other.endDate).max)
    } else {
      this
    }
  }
}
