package com.xebialabs.xlrelease.domain.delivery.conditions;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.delivery.Condition;
import com.xebialabs.xlrelease.domain.delivery.Delivery;
import com.xebialabs.xlrelease.domain.delivery.TrackedItemStatus;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static java.util.stream.Collectors.toList;

@Metadata(versioned = false, description = "Condition which completes when all tracked items preceding the transition are in a certain status.")
@PublicApiRef
@ShowOnlyPublicApiMembers
public class ItemsCompletionCondition extends Condition {
    private static final List<String> DEFAULT_COMPLETED_STATUSES = Arrays.stream(TrackedItemStatus.DONE_STATUSES).map(Objects::toString).collect(toList());

    @Property(isTransient = true, hidden = true, defaultValue = "READY,SKIPPED", description = "Allowed statuses of tracked items.")
    private List<String> statuses = DEFAULT_COMPLETED_STATUSES;

    @Property(isTransient = true, hidden = true, defaultValue = "true", description = "Transition even with de-scoped tracked items present.")
    private boolean ignoreDescoped = true;

    // --

    public ItemsCompletionCondition() {
    }

    public ItemsCompletionCondition(final List<String> statuses) {
        this.statuses = statuses;
    }

    @Override
    public String getDescription() {
        return "Wait for all items to be completed";
    }

    @Override
    public void validate(Delivery delivery) {
        checkNotNull(statuses, "Statuses");
        checkArgument(statuses.size() > 0, "Statuses must not be empty");
        checkArgument(DEFAULT_COMPLETED_STATUSES.containsAll(statuses), "Status must be one of the values: %s", DEFAULT_COMPLETED_STATUSES);
    }

    // --

    public List<String> getStatuses() {
        return statuses;
    }

    public void setStatuses(final List<String> statuses) {
        this.statuses = statuses;
    }

    public boolean isIgnoreDescoped() {
        return ignoreDescoped;
    }

    public void setIgnoreDescoped(final boolean ignoreDescoped) {
        this.ignoreDescoped = ignoreDescoped;
    }
}
