package com.xebialabs.xlrelease.domain.events

import com.xebialabs.xlrelease.domain.Phase

/**
  * Common interface for domain events related to phase execution or modification.
  */
sealed trait PhaseEvent extends XLReleaseEvent

//
// Phase CUD
//

/**
  * Event published when a phase is created.
  * @param phase the created phase.
  */
case class PhaseCreatedEvent(phase: Phase) extends PhaseEvent

/**
  * Event published when a phase is copied.
  * @param phaseCopy copied instance of the phase.
  */
case class PhaseCopiedEvent(phaseCopy: Phase) extends PhaseEvent

/**
  * Event published when a phase is duplicated.
  * @param phaseDuplicate duplicated instance of the phase.
  */
case class PhaseDuplicatedEvent(phaseDuplicate: Phase) extends PhaseEvent

/**
  * Event published when a phase is moved from one position to another within a release or a template.
  * @param movedPhase the instance of the phase which was moved.
  * @param originIndex the original index of the phase in the release or template, 0-based.
  * @param targetIndex the new index of the phase in the release or template, 0-based.
  */
case class PhaseMovedEvent(movedPhase: Phase, originIndex: Int, targetIndex: Int) extends PhaseEvent

/**
  * Event published when properties or a phase are updated.
  * @param original a copy of the phase before it was updated.
  * @param updated the updated copy of the phase. You can use [[original]] to compare which properties have changed.
  */
case class PhaseUpdatedEvent(original: Phase, updated: Phase) extends PhaseEvent

/**
  * Event published when a phase is deleted.
  * @param phase a copy of the deleted release or template. Note that when this event is
  *              handled the phase does not exist in the repository anymore.
  */
case class PhaseDeletedEvent(phase: Phase) extends PhaseEvent


//
// Phase execution
//

/**
  * Common interface for events when the phase status changes.
  */
sealed trait PhaseExecutionEvent extends PhaseEvent {
  /**
    * The phase of which status has changed.
    */
  val phase: Phase
}

/**
  * Event published when a phase is started.
  * @param phase instance of the phase.
  */
case class PhaseStartedEvent(phase: Phase) extends PhaseExecutionEvent

/**
  * Event published when a phase has failed.
  * @param phase instance of the phase.
  */
case class PhaseFailedEvent(phase: Phase) extends PhaseExecutionEvent

/**
  * Event published when a phase goes into `FAILING` state. This happens when a parallel group is active and it
  * has both `IN_PROGRESS` and `FAILED` tasks.
  * @param phase instance of the phase.
  */
case class PhaseStartedFailingEvent(phase: Phase) extends PhaseExecutionEvent

/**
  * Event published when one of phase's tasks is retried. More specifically it happens when a phase goes
  * from `FAILED` or `FAILING` state into `IN_PROGRESS` state.
  * @param phase instance of the phase.
  */
case class PhaseRetriedEvent(phase: Phase) extends PhaseExecutionEvent

/**
  * Event published when a phase is completed.
  * @param phase instance of the phase.
  */
case class PhaseCompletedEvent(phase: Phase) extends PhaseExecutionEvent

/**
  * Event published when a phase is closed and restarted. When closing a phase all its unfinished tasks are skipped.
  * @param phase instance of the phase.
  */
case class PhaseClosedEvent(phase: Phase) extends PhaseExecutionEvent
