package com.xebialabs.xlrelease.domain.events

import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.domain.variables.Variable

import java.util.{List => JList}

/**
 * Common interface for domain events related to variables modification.
 */
sealed trait VariableEvent extends XLReleaseEvent

sealed trait VariableModificationSource

case class ModifiedFromScriptTask(taskId: String) extends VariableModificationSource

//
// Release variables
//

/**
  * Common interface for domain events related to release variables.
  */
sealed trait ReleaseVariableEvent extends VariableEvent

/**
  * Event published when a variable is created in a release.
  *
  * @param variable created variable.
  */
case class ReleaseVariableCreatedEvent(variable: Variable) extends ReleaseVariableEvent

/**
  * Event published when a variable is updated in a release.
  *
  * @param original a copy of the original variable.
  * @param updated  the updated variable. You can use [[original]] to find the changes.
  */
case class ReleaseVariableUpdatedEvent(original: Variable, updated: Variable) extends ReleaseVariableEvent

/**
  * Event published when a variable is deleted from a release.
  *
  * @param variable deleted variable.
  */
case class ReleaseVariableDeletedEvent(variable: Variable) extends ReleaseVariableEvent

/**
  * Event published when a variable has been replaced by another variable or a value. The first case is effectively
  * a rename of a variable, and the second is effectively a delete of variable's usages. If a variable is being deleted
  * then this event will precede the [[ReleaseVariableDeletedEvent]].
  *
  * @param variable    a copy of the original variable.
  * @param replacement the replacement of the variable: it contains either a new variable name or a value with which it was replaced.
  */
case class ReleaseVariableReplacedEvent(variable: Variable, replacement: VariableOrValue) extends ReleaseVariableEvent

/**
  * Event published when a bulk update to release variables has happened. Such an update happens usually after a script task execution
  * in which the `releaseVariables` dictionary has changed. This update can lead to variables being created, updated and deleted.
  * Note that when this event is published no specific single variable related events are published.
  *
  * @param originalVariables the list of release variables before the update.
  * @param updatedVariables  the list of release variables after the update.
  */
case class ReleaseVariablesUpdatedEvent(originalVariables: JList[Variable], updatedVariables: JList[Variable]) extends ReleaseVariableEvent

/**
 * Event published when template variables are changed and need to be synced.
 *
 * @param templateId        template id
 * @param templateVariables a mutable map of template variables
 */
case class TemplateVariablesChangedEvent(templateId: String, templateVariables: JList[Variable]) extends XLReleaseEvent

//
// Global variables
//

/**
  * Common interface for domain events related to global variables.
  */
sealed trait GlobalVariableEvent extends VariableEvent

/**
  * Event published when a global variable is created. Currently this event is limited only to situation when the global
  * variable is created from a script task in a release, using `globalVariables` dictionary.
  *
  * @param variable created global variable.
  * @param source   the way it was created. Currently only [[ModifiedFromScriptTask]] is supported.
  */
case class GlobalVariableCreatedEvent(variable: Variable, source: VariableModificationSource) extends GlobalVariableEvent

/**
  * Event published when a global variable is updated. Currently this event is limited only to situation when the global
  * variable is updated from a script task in a release, using `globalVariables` dictionary.
  *
  * @param original a copy of the original global variable.
  * @param updated  the updated global variable. As this can only be triggered from a script task, only the variable value can be different.
  * @param source   the way it was updated. Currently only [[ModifiedFromScriptTask]] is supported.
  */
case class GlobalVariableUpdatedEvent(original: Variable, updated: Variable, source: VariableModificationSource) extends GlobalVariableEvent

/**
  * Event published when a global variable is deleted. Currently this event is limited only to situation when the global
  * variable is created from a script task in a release, using `globalVariables` dictionary.
  *
  * @param variable deleted global variable.
  * @param source   the way it was deleted. Currently only [[ModifiedFromScriptTask]] is supported.
  */
case class GlobalVariableDeletedEvent(variable: Variable, source: VariableModificationSource) extends GlobalVariableEvent


//
// Folder variables
//

/**
  * Common interface for domain events related to folder variables.
  */
sealed trait FolderVariableEvent extends VariableEvent

/**
  * Event published when a folder variable is created.
  *
  * @param variable created folder variable.
  * @param source   the way it was created. Currently only [[ModifiedFromScriptTask]] is supported.
  */
case class FolderVariableCreatedEvent(variable: Variable, source: VariableModificationSource) extends FolderVariableEvent

/**
  * Event published when a folder variable is updated.
  *
  * @param original a copy of the original folder variable.
  * @param updated  the updated folder variable. As this can only be triggered from a script task, only the variable value can be different.
  * @param source   the way it was created. Currently only [[ModifiedFromScriptTask]] is supported.
  */
case class FolderVariableUpdatedEvent(original: Variable, updated: Variable, source: VariableModificationSource) extends FolderVariableEvent

/**
  * Event published when a folder variable is deleted.
  *
  * @param variable deleted folder variable.
  * @param source   the way it was created. Currently only [[ModifiedFromScriptTask]] is supported.
  */
case class FolderVariableDeletedEvent(variable: Variable, source: VariableModificationSource) extends FolderVariableEvent
