package com.xebialabs.xlrelease.domain.status;

import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;

@PublicApiRef
@ShowOnlyPublicApiMembers
public enum TaskStatus {
    /**
     * When a task is not started yet,
     */
    PLANNED("planned"),
    /**
     * When a task is about to be started, but is waiting for its scheduled start date to be reached.
     */
    PENDING("pending"),
    /**
     * When a task has been started
     */
    IN_PROGRESS("in_progress"),
    /**
     * @deprecated Used only in upgrader to deserialize release
     */
    @Deprecated(since = "10.2.0", forRemoval = true)
    QUEUED("queued"),
    /**
     * @deprecated Used only in upgrader to deserialize release
     */
    @Deprecated(since = "10.2.0", forRemoval = true)
    ABORT_SCRIPT_QUEUED("abort_script_queued"),
    /**
     * @deprecated Used only in upgrader to deserialize release
     */
    @Deprecated(since = "10.2.0", forRemoval = true)
    FAILURE_HANDLER_QUEUED("failure_handler_queued"),
    /**
     * When a task is done
     */
    COMPLETED("completed"),
    /**
     * When a task that is not yet in progress is completed
     */
    COMPLETED_IN_ADVANCE("completed_in_advance"),
    /**
     * When a task is done, but completion was forced.
     */
    SKIPPED("skipped"),
    /**
     * When a task that is not yet in progress is skipped
     */
    SKIPPED_IN_ADVANCE("skipped_in_advance"),
    /**
     * When a task has been failed, it can be re-tried.
     */
    FAILED("failed"),
    /**
     * When a parallel group has failed subtasks, but other subtasks still in progress or pending.
     */
    FAILING("failing"),
    /**
     * When the release is aborted, all tasks that are not completed are marked as aborted.
     */
    ABORTED("aborted"),
    /**
     * When a task's precondition is being evaluated
     */
    PRECONDITION_IN_PROGRESS("precondition_in_progress"),
    /**
     * When a task needs extra input to be started: missing required variables.
     */
    WAITING_FOR_INPUT("waiting_for_input"),
    /**
     * When a task's failure handler is being evaluated
     */
    FAILURE_HANDLER_IN_PROGRESS("failure_handler_in_progress"),
    /**
     * When the facet (aka attributes are being checked)
     */
    FACET_CHECK_IN_PROGRESS("facet_check_in_progress"),
    /**
     * When a task's abort script is being evaluated
     */
    ABORT_SCRIPT_IN_PROGRESS("abort_script_in_progress");

    public static final TaskStatus[] ACTIVE_STATUSES = new TaskStatus[]{
            PENDING,
            PRECONDITION_IN_PROGRESS,
            FAILURE_HANDLER_IN_PROGRESS,
            ABORT_SCRIPT_IN_PROGRESS,
            IN_PROGRESS,
            FAILING,
            FAILED,
            WAITING_FOR_INPUT
    };

    public static final TaskStatus[] IN_PROGRESS_STATUSES = new TaskStatus[]{
            PRECONDITION_IN_PROGRESS,
            IN_PROGRESS,
            FAILURE_HANDLER_IN_PROGRESS,
            FACET_CHECK_IN_PROGRESS,
            WAITING_FOR_INPUT
    };

    public static final TaskStatus[] FAILED_STATUSES = new TaskStatus[]{FAILING, FAILED};

    private final String value;

    TaskStatus(String v) {
        value = v;
    }

    public String value() {
        return value;
    }

    public boolean hasBeenStarted() {
        return this != PLANNED && this != PENDING && this != WAITING_FOR_INPUT;
    }

    public boolean isDone() {
        return isOneOf(COMPLETED, SKIPPED);
    }

    public boolean isDoneInAdvance() {
        return isOneOf(COMPLETED_IN_ADVANCE, SKIPPED_IN_ADVANCE);
    }

    public boolean isActive() {
        return isOneOf(ACTIVE_STATUSES);
    }

    public boolean isInactive() {
        return !isActive();
    }

    public boolean isOneOf(TaskStatus... statuses) {
        for (TaskStatus status : statuses) {
            if (this == status) {
                return true;
            }
        }

        return false;
    }
}
