package com.xebialabs.xlrelease.domain.variables;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import com.google.common.base.Joiner;

import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.variables.Variable.VariableWithValue;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

@PublicApiRef
@ShowOnlyPublicApiMembers
public class ListStringVariable extends VariableWithValue<List<String>> {

    @PublicApiMember
    @Property(required = false, description = "Value of the release variable or default value of the template variable")
    protected List<String> value;

    @PublicApiMember
    @Override
    public List<String> getValue() {
        return value == null ? getEmptyValue() : newArrayList(value);
    }

    @Override
    public List<String> getEmptyValue() {
        return new ArrayList<>();
    }

    @Override
    public boolean isValueEmpty() {
        return value == null || value.isEmpty();
    }

    @PublicApiMember
    @Override
    public void setValue(final List<String> value) {
        if(value == null) {
            this.value = getEmptyValue();
        }
        else {
            this.value = value;
        }
    }

    @Override
    public void setUntypedValue(Object newValue) {
        if (newValue == null) {
            this.setValue(null);
        } else if (newValue instanceof Collection) {
            @SuppressWarnings("unchecked")
            Collection<Object> items = (Collection<Object>) newValue;
            List<String> list = items.stream().map(VariableHelper::toString).collect(Collectors.toList());
            this.setValue(list);
        } else {
            throw new IllegalArgumentException(format("Cannot set instance of %s into list of strings variable value: %s",
                    newValue.getClass().getSimpleName(), newValue));
        }
    }

    @Override
    public boolean isPassword() {
        return false;
    }

    @Override
    public String getValueAsString() {
        return formatValueAsString(getValue());
    }

    @Override
    public String getEmptyValueAsString() {
        return formatValueAsString(getEmptyValue());
    }

    @Override
    public boolean isValueAssignableFrom(final Object value) {
        return value instanceof Collection || value instanceof Iterable;
    }

    private String formatValueAsString(final List<String> valueToFormat) {
        List<String> withQuotes = valueToFormat
                .stream()
                .map(e -> String.format("'%s'", escapeQuotes(e)))
                .collect(Collectors.toList());
        return format("[%s]", Joiner.on(", ").join(withQuotes));
    }
}
