package com.xebialabs.xlrelease.domain;

import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;

/**
 * The Team domain object in Digital.ai Release is a transient object and is not saved into the repository.
 * It is kept for backwards-compatibility, as it was part of the public API.
 * The teams' data is stored by the platform using the {@link com.xebialabs.deployit.security.RoleService} and the
 * platform's own {@link com.xebialabs.deployit.security.Role} object.
 */
@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(versioned = false)
public class Team extends BaseConfigurationItem implements TeamPermissionOperations {

    public static final String FOLDER_OWNER_TEAMNAME = "Folder Owner";
    public static final String TEMPLATE_OWNER_TEAMNAME = "Template Owner";
    public static final String RELEASE_ADMIN_TEAMNAME = "Release Admin";

    @Property
    private String teamName;

    @Property(required = false)
    private List<String> members = new ArrayList<>();

    @Property(required = false)
    private List<String> roles = new ArrayList<>();

    @Property(required = false)
    private List<String> permissions = new ArrayList<>();

    @Override
    @PublicApiMember
    public String getTeamName() {
        return teamName;
    }

    public void setTeamName(String teamName) {
        this.teamName = teamName;
    }

    public boolean hasMember(String username) {
        return members.stream().anyMatch(member -> member.equalsIgnoreCase(username));
    }

    public boolean hasAnyMember(Collection<String> members) {
        return members.stream().anyMatch(this::hasMember);
    }

    @Override
    public boolean hasPermission(String permission) {
        return permissions.contains(permission);
    }

    @PublicApiMember
    public List<String> getMembers() {
        return members;
    }

    public void setMembers(List<String> members) {
        this.members = new ArrayList<>(members);
    }

    public void addMember(String newMember) {
        for (String member : members) {
            if (member.equals(newMember)) return;
        }
        if (newMember != null) {
            members.add(newMember);
        }
    }

    @PublicApiMember
    public List<String> getRoles() {
        return roles;
    }

    public void setRoles(List<String> roles) {
        this.roles = new ArrayList<>(roles);
    }

    public void addRole(String newRole) {
        for (String role : roles) {
            if (role.equals(newRole)) return;
        }
        if (newRole != null) {
            roles.add(newRole);
        }
    }

    public boolean hasRole(String role) {
        return roles.contains(role);
    }

    public boolean hasAnyRole(List<Role> roles) {
        for (Role role : roles) {
            if (hasRole(role.getName())) return true;
        }
        return false;
    }

    @PublicApiMember
    public List<String> getPermissions() {
        return permissions;
    }

    public void setPermissions(final List<String> permissions) {
        this.permissions = permissions;
    }

    public void addPermission(String permission) {
        if (!permissions.contains(permission) && permission != null) {
            permissions.add(permission);
        }
    }

    public void addMembers(final String[] members) {
        for (String member : members) {
            addMember(member);
        }
    }

    public void addRoles(final String[] roles) {
        for (String role : roles) {
            addRole(role);
        }
    }

    public void addPermissions(final String[] permissions) {
        for (String permission : permissions) {
            addPermission(permission);
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!Team.class.isAssignableFrom(obj.getClass())) {
            return false;
        }
        final Team other = (Team) obj;

        final List<String> otherMembers = other.getMembers().stream().map(String::toLowerCase).collect(Collectors.toList());
        final List<String> thisMembers = members.stream().map(String::toLowerCase).collect(Collectors.toList());

        return ((other.getId() == null && id == null) || (other.getId() != null && other.getId().equals(id))) &&
                ((other.getTeamName() == null && teamName == null) || (other.getTeamName() != null && other.getTeamName().equals(teamName))) &&
                other.getPermissions().containsAll(permissions) && permissions.containsAll(other.getPermissions()) &&
                otherMembers.containsAll(thisMembers) && thisMembers.containsAll(otherMembers) &&
                other.getRoles().containsAll(roles) && roles.containsAll(other.getRoles());
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, teamName, permissions, members, roles);
    }

    @Override
    public String toString() {
        return "{id: '" + id + "', teamName: '" + teamName + "', members: " + Arrays.toString(members.toArray()) + ", roles: " + Arrays.toString(roles.toArray()) + ", permissions: " + Arrays.toString(permissions.toArray()) + "}";
    }

    public static boolean isSystemTeam(String teamName) {
        Team t = new Team();
        t.setTeamName(teamName);
        return t.isSystemTeam();
    }

}
