package com.xebialabs.xlrelease.domain.runner;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.xebialabs.deployit.plugin.api.validation.ExtendedValidationContext;
import com.xebialabs.deployit.plugin.api.validation.Rule;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.Validator;

@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = RegistrySettingsValidator.RunnerRegistrySettingsValidator.class, type = "remoteRunnerRegistrySettings")
@Target(ElementType.TYPE)
@interface RegistrySettingsValidator {
    class RunnerRegistrySettingsValidator implements Validator<RemoteRunnerRegistrySettings> {

        @Override
        public void validate(RemoteRunnerRegistrySettings remoteRunnerRegistrySettings, ValidationContext context) {
            ExtendedValidationContext extendedContext = (ExtendedValidationContext) context;

            validateBaseUrlAtBeginningOfMapping(remoteRunnerRegistrySettings, extendedContext);
            validateImageMappingsAreNotTheSame(remoteRunnerRegistrySettings, extendedContext);
            validateMappingsDontStartOrEndWithSlash(remoteRunnerRegistrySettings, extendedContext);
        }

        private void validateBaseUrlAtBeginningOfMapping(RemoteRunnerRegistrySettings remoteRunnerRegistrySettings, ExtendedValidationContext extendedContext) {
            Set<String> mappings = remoteRunnerRegistrySettings.imageUrlMappings().entrySet().stream()
                    .flatMap(entry -> Stream.of(entry.getKey(), entry.getValue()))
                    .collect(Collectors.toSet());

            for (String baseUrl : remoteRunnerRegistrySettings.getBaseUrls()) {
                boolean isValid = false;
                for (String mapping : mappings) {
                    if (mapping.startsWith(baseUrl)) {
                        isValid = true;
                        break;
                    }
                }
                if (!isValid) {
                    extendedContext.error("Base URL '%s' must be an beginning of at least one image URL mapping", baseUrl);
                }
            }
        }

        private void validateImageMappingsAreNotTheSame(RemoteRunnerRegistrySettings remoteRunnerRegistrySettings, ExtendedValidationContext extendedContext) {
            remoteRunnerRegistrySettings.imageUrlMappings().entrySet().stream()
                    .filter(entry -> entry.getKey().equals(entry.getValue()))
                    .forEach(entry -> extendedContext.error("Image URL mapping '%s' cannot map to itself", entry.getKey()));
        }

        private void validateMappingsDontStartOrEndWithSlash(RemoteRunnerRegistrySettings remoteRunnerRegistrySettings, ExtendedValidationContext extendedContext) {
            remoteRunnerRegistrySettings.imageUrlMappings().entrySet().stream()
                    .flatMap(entry -> Stream.of(entry.getKey(), entry.getValue()))
                    .filter(mapping -> mapping.startsWith("/") || mapping.endsWith("/"))
                    .forEach(entry -> extendedContext.error("Image URL mapping '%s' cannot start or end with a '/' character", entry));
        }
    }
}