package com.xebialabs.xlrelease.domain.variables;

import java.util.Collection;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import com.google.common.base.Joiner;

import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.variables.Variable.VariableWithValue;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Sets.newHashSet;
import static java.lang.String.format;

@PublicApiRef
@ShowOnlyPublicApiMembers
public class SetStringVariable extends VariableWithValue<Set<String>> {

    @PublicApiMember
    @Property(required = false, description = "Value of the release variable or default value of the template variable")
    protected Set<String> value;

    @PublicApiMember
    @Override
    public Set<String> getValue() {
        return value == null ? getEmptyValue() : newHashSet(value);
    }

    @Override
    public Set<String> getEmptyValue() {
        return newHashSet();
    }

    @Override
    public boolean isValueEmpty() {
        return value == null || value.isEmpty();
    }

    @PublicApiMember
    @Override
    public void setValue(final Set<String> value) {
        if (value == null) {
            this.value = getEmptyValue();
        } else {
            this.value = value;
        }
    }

    @Override
    public void setUntypedValue(Object newValue) {
        if (newValue == null) {
            this.setValue(null);
        } else if (newValue instanceof Collection) {
            @SuppressWarnings("unchecked")
            Collection<Object> items = (Collection<Object>) newValue;
            Set<String> set = items.stream().map(VariableHelper::toString).collect(Collectors.toSet());
            this.setValue(set);
        } else {
            throw new IllegalArgumentException(format("Cannot set instance of %s into Set of strings variable value: %s",
                    newValue.getClass().getSimpleName(), newValue));
        }
    }

    @Override
    public boolean isPassword() {
        return false;
    }

    @Override
    public String getValueAsString() {
        Set<String> withQuotes = getValue()
                .stream()
                .map(e -> String.format("'%s'", escapeQuotes(e)))
                .collect(Collectors.toSet());
        return format("set([%s])", Joiner.on(", ").join(new TreeSet<>(withQuotes)));
    }

    @Override
    public String getEmptyValueAsString() {
        return "set()";
    }

    @Override
    public boolean isValueAssignableFrom(final Object value) {
        return value instanceof Collection;
    }

}
