package com.xebialabs.xlrelease.serialization.json.xltype;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.*;
import javax.json.*;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.udm.ExternalProperty;
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xlrelease.serialization.json.utils.JsonWithCachedProvider;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.SerializationException;
import com.xebialabs.xltype.serialization.util.DateUtil;

import static javax.json.JsonValue.ValueType.OBJECT;
import static java.lang.String.format;

public class CiJson2Reader implements CiReader {

    private final JsonObject json;

    private final Iterator<String> propertyIterator;
    private String currentPropertyName;

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    public CiJson2Reader(JsonObject json) {
        this.json = json;
        this.propertyIterator = properties(json).iterator();
    }

    public static CiJson2Reader create(String jsonObject) {
        try (Reader reader = new StringReader(jsonObject);
             JsonReader jsonReader = JsonWithCachedProvider.createReader(reader)) {
            return new CiJson2Reader(jsonReader.readObject());
        } catch (IOException | JsonException e) {
            throw new IllegalArgumentException("Can't parse the following as a JSON object:\n" + jsonObject, e);
        }
    }

    //
    // CiReader implementation
    //

    @Override
    public String getType() {
        try {
            return json.getString("type");
        } catch (Exception e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getId() {
        try {
            if (json.isNull("id")) {
                return null;
            } else {
                return json.getString("id");
            }
        } catch (Exception e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getToken() {
        return getStringOrNull("$token");
    }

    @Override
    public CiAttributes getCiAttributes() {
        final String createdBy = getStringOrNull("$createdBy");
        final DateTime createdAt = getDateTimeOrNull("$createdAt");
        final String lastModifiedBy = getStringOrNull("$lastModifiedBy");
        final DateTime lastModifiedAt = getDateTimeOrNull("$lastModifiedAt");
        final String unparsedSCMData = getStringOrNull("$scmTraceabilityDataId");
        Integer scmData = null;

        if (unparsedSCMData != null) {
            scmData = Integer.parseInt(unparsedSCMData);
        }

        return new CiAttributes(createdBy, createdAt, lastModifiedBy, lastModifiedAt, scmData);
    }

    @Override
    public boolean hasMoreProperties() {
        return propertyIterator.hasNext();
    }

    @Override
    public void moveIntoProperty() {
        currentPropertyName = propertyIterator.next();
    }

    @Override
    public CiReader moveIntoNestedProperty() {
        try {
            return new CiJson2Reader(json.getJsonObject(currentPropertyName));
        } catch (Exception e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public void moveOutOfProperty() {
        // Noop
    }

    @Override
    public String getCurrentPropertyName() {
        return currentPropertyName;
    }

    @Override
    public String getStringValue() {
        try {
            JsonValue jsonValue = json.get(currentPropertyName);
            switch (jsonValue.getValueType()) {
                case NULL:
                    return null;
                case TRUE:
                    return "true";
                case FALSE:
                    return "false";
                case NUMBER:
                    return json.getJsonNumber(currentPropertyName).toString();
                case STRING:
                default:
                    return json.getString(currentPropertyName);
            }
        } catch (Exception e) {
            logger.error(format("Incorrect property value type or value for property '%s'", currentPropertyName));
            throw new SerializationException(e);
        }
    }

    @Override
    public List<String> getStringValues() {
        try {
            JsonArray strings = json.getJsonArray(currentPropertyName);
            return toList(strings);
        } catch (Exception e) {
            logger.error(format("[Property '%s'] Incorrect property value type Array or value", currentPropertyName));
            throw new SerializationException(e);
        }
    }

    @Override
    public Map<String, String> getStringMap() {
        try {
            JsonObject map = json.getJsonObject(currentPropertyName);
            return toMap(map);
        } catch (Exception e) {
            logger.error(format("[Property '%s'] Incorrect value type Object or value", currentPropertyName));
            throw new SerializationException(e);
        }
    }

    @Override
    public boolean isCiReference() {
        JsonValue propertyValue = json.get(currentPropertyName);
        return currentPropertyName != null && (propertyValue == null || propertyValue.getValueType() != OBJECT);
    }

    @Override
    public String getCiReference() {
        return getStringValue();
    }

    @Override
    public List<String> getCiReferences() {
        return getStringValues();
    }

    @Override
    public CiListReader getCurrentCiListReader() {
        try {
            return new CiListJson2Reader(json.getJsonArray(currentPropertyName));
        } catch (Exception e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<ValidationMessage> getValidationMessages() {
        List<ValidationMessage> messages = new ArrayList<>();
        try {
            JsonArray array = json.getJsonArray("validation-messages");
            for (int i = 0; i < array.size(); i++) {
                messages.add(toMessage(array.getJsonObject(i)));
            }
        } catch (Exception e) {
            throw new SerializationException(e);
        }
        return messages;
    }

    @Override
    public Map<String, ExternalProperty> getExternalProperties() {
        try {
            HashMap<String, ExternalProperty> result = new HashMap<>();
            JsonObject object = json.getJsonObject("external-properties");
            for (String key : object.keySet()) {
                JsonObject jsonObject = object.getJsonObject(key);
                String kind = jsonObject.getString("kind");
                if (kind.equals("lookup")) {
                    LookupValueKey lookupValueKey = new LookupValueKey();
                    lookupValueKey.setKey(jsonObject.getString("key"));
                    lookupValueKey.setProviderId(jsonObject.getString("provider"));
                    result.put(key, lookupValueKey);
                } else {
                    throw new IllegalArgumentException("Unknown external property kind: '" + kind + "'.");
                }
            }
            return result;
        } catch (JsonException e) {
            throw new SerializationException(e);
        }
    }

    //
    // Util
    //

    private static Collection<String> properties(JsonObject object) {
        List<String> properties = new ArrayList<>();
        for (String key : object.keySet()) {
            if (!isSpecialKey(key)) {
                properties.add(key);
            }
        }

        return properties;
    }

    private static boolean isSpecialKey(String key) {
        return "id".equals(key) || "type".equals(key) || (key != null && key.startsWith("$"));
    }

    private static List<String> toList(JsonArray array) {
        List<String> strings = new ArrayList<>(array.size());
        for (int i = 0; i < array.size(); i++) {
            if (!array.isNull(i)) {
                strings.add(array.getString(i));
            }
        }

        return strings;
    }

    private static Map<String, String> toMap(JsonObject object) {
        Map<String, String> map = new LinkedHashMap<>();
        for (String key : object.keySet()) {
            map.put(key, String.valueOf(getStringOrNull(object, key)));
        }

        return map;
    }

    private static ValidationMessage toMessage(JsonObject jsonObject) {
        return new ValidationMessage(
                jsonObject.getString("ci"),
                getStringOrNull(jsonObject, "property"),
                jsonObject.getString("message"),
                getStringOrNull(jsonObject, "level"));
    }

    private String getStringOrNull(String key) {
        return getStringOrNull(json, key);
    }

    private static String getStringOrNull(JsonObject json, String key) {
        try {
            if (json.containsKey(key)) {
                if (json.isNull(key)) {
                    return null;
                }
                return json.getString(key);
            }
        } catch (JsonException e) {
            throw new SerializationException(e);
        }
        return null;
    }

    private DateTime getDateTimeOrNull(String key) {
        String stringOrNull = getStringOrNull(key);
        if (stringOrNull == null) {
            return null;
        }
        return DateUtil.fromString(stringOrNull);
    }
}
