package com.xebialabs.xlrelease.serialization.json.xltype;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.serialization.json.repository.ResolverRepository;
import com.xebialabs.xlrelease.utils.Collectors;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.CiReference;

/**
 * CiConverter which does or does not decrypt the passwords read from the json based on encrypter.
 * <p>
 * Use NonDecryptingPasswordEncrypter to not decrypt the password
 * Use NullPasswordEncrypter to keep value for password as it is.
 */
public class XlrPasswordEncryptingCiConverter extends PasswordEncryptingCiConverter {

    private static final Logger logger = LoggerFactory.getLogger(XlrPasswordEncryptingCiConverter.class);

    private final PasswordEncrypter passwordEncrypter;
    private Map<String, ConfigurationItem> folderLessIdToItem = new HashMap<>();

    private ResolveOptions resolveOptions = ResolveOptions.WITH_DECORATORS();

    public XlrPasswordEncryptingCiConverter(PasswordEncrypter passwordEncrypter) {
        this.passwordEncrypter = passwordEncrypter;
    }

    public XlrPasswordEncryptingCiConverter(PasswordEncrypter passwordEncrypter, List<ConfigurationItem> targetConfigurationItems, ResolveOptions resolveOptions) {
        this.passwordEncrypter = passwordEncrypter;
        this.folderLessIdToItem = targetConfigurationItems.stream()
                .filter(i -> i.getId() != null)
                .collect(Collectors.toMap(i -> Ids.getFolderlessId(i.getId()), Function.identity()));
        this.resolveOptions = resolveOptions;
    }

    @Override
    protected ConfigurationItem createConfigurationItem(CiReader reader) {
        String id = reader.getId();
        if (id != null) {
            String folderlessId = Ids.getFolderlessId(id);
            if (folderLessIdToItem.containsKey(folderlessId)) {
                return folderLessIdToItem.get(folderlessId);
            }
        }
        return super.createConfigurationItem(reader);
    }

    @Override
    protected void readProperty(CiReader reader, Descriptor descriptor, ConfigurationItem configurationItem) {
        try {
            super.readProperty(reader, descriptor, configurationItem);
        } catch (Exception ex) {
            logger.warn("Unable to read property [{}] of type [{}]: {}.", reader.getCurrentPropertyName(), descriptor.getType(), ex.getMessage());
        }
    }

    @Override
    protected void readStringProperty(final ConfigurationItem configurationItem, final PropertyDescriptor propertyDescriptor, final CiReader reader) {
        if (propertyDescriptor.isPassword()) {
            propertyDescriptor.set(configurationItem, passwordEncrypter.ensureDecrypted(reader.getStringValue()));
        } else {
            super.readStringProperty(configurationItem, propertyDescriptor, reader);
        }
    }


    @Override
    public void resolveReferences(final Repository repository) {
        for (CiReference reference : getReferences()) {

            // Resolve IDs to CIs
            List<ConfigurationItem> resolvedCIs = new ArrayList<>();

            for (String id : reference.getIds()) {
                ConfigurationItem alsoRead = getReadCIs().get(id);
                if (alsoRead == null && Ids.isInRelease(id)) { // make sure we do find items in a release
                    String folderLessId = Ids.getFolderlessId(id);
                    alsoRead = getReadCIs().get(folderLessId);
                }
                if (alsoRead != null) {
                    logger.trace("Resolved reference [{}] for [{}] with an already provided reference", id, reference.getCi().getId());
                    resolvedCIs.add(alsoRead);
                } else {
                    if (resolveOptions.shouldResolveReferences()) {
                        resolveFromRepository(repository, reference, resolvedCIs, id);
                    }
                }
            }

            // Set referred CIs on the parsed CI
            reference.set(resolvedCIs);
        }
    }

    private void resolveFromRepository(final Repository repository, final CiReference reference, final List<ConfigurationItem> resolvedCIs, final String id) {
        if (repository instanceof ResolverRepository) {
            // already read cis and folderLessIdToItem map are connected
            ConfigurationItem cachedItem = folderLessIdToItem.get(id);
            if (null == cachedItem) {
                ResolverRepository resolver = (ResolverRepository) repository;
                ConfigurationItem resolvedItem = resolver.resolve(id, reference);
                if (resolvedItem == null) {
                    // it could happen that target points to an archived item - and we do not resolve those
                    logger.warn("Unable to resolve reference [{}] to [{}]", reference, id);
                } else {
                    logger.trace("Resolved reference [{}] for [{}]", id, reference.getCi().getId());
                    resolvedCIs.add(resolvedItem);
                    addFolderLessItem(resolvedItem);
                }
            } else {
                resolvedCIs.add(cachedItem);
            }
        } else {
            throw new IllegalStateException("Unable to resolve reference because repository is not ResolverRepository");
        }
    }

    private void addFolderLessItem(ConfigurationItem resolvedItem) {
        String folderLessId = Ids.getFolderlessId(resolvedItem.getId());
        folderLessIdToItem.put(folderLessId, resolvedItem);
    }
}
