package com.xebialabs.xlrelease.support.pekko.spring

import grizzled.slf4j.Logging
import org.apache.pekko.actor.{Actor, ActorRef, ActorRefFactory, ExtendedActorSystem, Extension, ExtensionId, Props}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.ApplicationContext

import java.beans.BeanProperty

// This class will be exposed as Pekko extension bean to Spring, but it has to be properly initialized by Spring before it's 1st use.
// Since Pekko is created AFTER Spring already registered this extension as proper spring bean it is safe to use SpringExtension object
// from WITHIN Pekko actors.
// BUT, if this extension is used inside Spring beans (actor factories etc.) make sure those depend on SpringExtension bean
// so that this extension would be properly initialized.
abstract class SpringExtension extends Extension with ExtensionId[SpringExtension] {
  @Autowired
  @BeanProperty
  val applicationContext: ApplicationContext = null

  private var actorSystem: ExtendedActorSystem = _

  def props(actorBeanName: String): Props = {
    // untyped
    Props.create(classOf[SpringActorProducer], applicationContext, actorBeanName)
  }

  def props(actorClass: Class[_ <: Actor], args: AnyRef*): Props = {
    Props.create(classOf[TypedSpringActorProducer], applicationContext, actorClass, args)
  }

  def actorOf(actorClass: Class[_ <: Actor], name: String = null, args: Seq[AnyRef] = Seq(), dispatcher: String= null): ActorRef = {
    if (actorSystem == null) {
      throw new IllegalStateException("Actor system did not create extension yet")
    }
    childActorOf(actorClass, name, args, dispatcher)(actorSystem)
  }

  def childActorOf(actorClass: Class[_ <: Actor], name: String = null, args: Seq[AnyRef] = Seq(), dispatcher: String= null)
                  (implicit actorFactory: ActorRefFactory): ActorRef = {
    if (actorSystem == null) {
      throw new IllegalStateException("Actor system did not create extension yet")
    }
    val actorProps = if (dispatcher == null) {
      props(actorClass, args:_*)
    } else {
      props(actorClass, args:_*).withDispatcher(dispatcher)
    }
    if (null == name) {
      actorFactory.actorOf(actorProps)
    } else {
      actorFactory.actorOf(actorProps, name)
    }
  }

  protected def setSystem(system: ExtendedActorSystem): SpringExtension = {
    this.actorSystem = system
    this
  }

  def getActorSystem: ExtendedActorSystem = this.actorSystem
}

object SpringExtension extends SpringExtension with Logging {
  override def createExtension(system: ExtendedActorSystem): SpringExtension = {
    logger.info("Creating Spring Pekko extension")
    SpringExtension.setSystem(system)
  }
}
