package com.xebialabs.xlrelease.security.sql

import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import com.xebialabs.xlrelease.security.sql.db.Ids.isGlobalId
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Caching}
import org.springframework.context.annotation.Conditional
import org.springframework.stereotype.Component

object SecurityCacheEviction {
  def evictRegex(onConfigurationItem: String): String =
    if (isGlobalId(Option(onConfigurationItem))) "regex:.*" else s"regex:$onConfigurationItem-.*"
}

@EventListener
@Component
@Conditional(value = Array(classOf[SecurityCacheConfigurationCondition]))
@CacheConfig(cacheManager = SECURITY_CACHE_MANAGER)
class SecurityCacheEviction extends Logging {
  @AsyncSubscribe
  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = "#event.onConfigurationItem?:'global'"),
    new CacheEvict(
      cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES),
      key = "T(com.xebialabs.xlrelease.security.sql.SecurityCacheEviction).evictRegex(#event.onConfigurationItem)")
  ))
  def onEvictRoles(event: EvictRolesEvent): Unit = {
    logger.debug(s"Expired role caches on [${event.onConfigurationItem}]")
  }

  @AsyncSubscribe
  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_PERMISSIONS), key = "#event.onConfigurationItem?:'global'"),
    new CacheEvict(cacheNames = Array(SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = "'regex:' + (#event.onConfigurationItem?:'global') + '-.*'")
  ))
  def onEvictPermissions(event: EvictPermissionsEvent): Unit = {
    logger.debug(s"Expired permission caches on [${event.onConfigurationItem}]")
  }

  @AsyncSubscribe
  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  def onEvictAll(event: EvictAllEvent): Unit = {
    logger.debug(s"Expired all security caches")
  }
}

case class EvictAllEvent() extends DistributedXLReleaseEvent
case class EvictRolesEvent(onConfigurationItem: String) extends DistributedXLReleaseEvent
case class EvictPermissionsEvent(onConfigurationItem: String) extends DistributedXLReleaseEvent
