package com.xebialabs.xlrelease.security.sql

import com.xebialabs.deployit.booter.local.utils.Strings.isEmpty
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.{Permissions, Role, RoleService}
import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import com.xebialabs.xlrelease.security.sql.db.Ids.{GLOBAL_SECURITY_ALIAS, isGlobalId, toDbId}
import com.xebialabs.xlrelease.security.sql.db.Tables.RolePrincipal
import com.xebialabs.xlrelease.security.sql.db.{Ids, Tables}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable, Caching}
import org.springframework.security.core.Authentication
import slick.dbio.DBIOAction.{failed, seq}
import slick.jdbc.JdbcProfile
import slick.lifted.{Query => SQuery}

import java.util
import scala.concurrent.ExecutionContext
import scala.jdk.CollectionConverters._

@CacheConfig(cacheManager = SECURITY_CACHE_MANAGER)
class CachingSqlRoleService(securityDatabase: Database, broadcastService: BroadcastService) extends SqlRoleService(securityDatabase) with Logging {

  @Cacheable(cacheNames = Array(SECURITY_ROLES), key = GLOBAL_KEY)
  override def getRoles(): util.List[Role] = super.getRoles()

  @Cacheable(cacheNames = Array(SECURITY_ROLES))
  override def getRoles(onConfigurationItem: String): util.List[Role] = super.getRoles(onConfigurationItem)

  @Cacheable(cacheNames = Array(SECURITY_USER_ROLES), key = "'global-' + #auth?.name")
  override def getRolesFor(auth: Authentication): util.List[Role] = super.getRolesFor(auth)

  @Cacheable(cacheNames = Array(SECURITY_USER_ROLES), key = "'global-' + #principal")
  override def getRolesFor(principal: String): util.List[Role] = super.getRolesFor(principal)

  @Cacheable(cacheNames = Array(SECURITY_PERMISSION_ROLES), key = "'global-' + #permission")
  override def getRolesForPermission(permission: String): util.List[Role] = super.getRolesForPermission(permission)

  @Cacheable(cacheNames = Array(SECURITY_ROLE_ASSIGNMENTS), key = GLOBAL_KEY)
  override def readRoleAssignments(): util.List[Role] = super.readRoleAssignments()

  @Cacheable(cacheNames = Array(SECURITY_ROLE_ASSIGNMENTS))
  override def readRoleAssignments(onConfigurationItem: String): util.List[Role] = super.readRoleAssignments(onConfigurationItem)

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true),
  ))
  override def writeRoleAssignments(roles: util.List[Role]): Unit = {
    super.writeRoleAssignments(roles)
    expireCache(GLOBAL_SECURITY_ALIAS)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def writeRoleAssignments(onConfigurationItem: String, roles: util.List[Role]): Unit = {
    super.writeRoleAssignments(onConfigurationItem, roles)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def createOrUpdateRole(role: Role, onConfigurationItem: String): String = {
    val roleId = super.createOrUpdateRole(role, onConfigurationItem)
    expireCache(onConfigurationItem)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def rename(name: String, newName: String, onConfigurationItem: String): String = {
    val roleId = super.rename(name, newName, onConfigurationItem)
    expireCache(onConfigurationItem)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def create(roles: Role*): Unit = {
    super.create(roles: _*)
    expireCache(GLOBAL_SECURITY_ALIAS)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def create(onConfigurationItem: String, roles: Role*): Unit = {
    super.create(onConfigurationItem, roles: _*)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def create(name: String, onConfigurationItem: String): String = {
    val roleId = super.create(name, onConfigurationItem)
    expireCache(onConfigurationItem)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def create(name: String): String = {
    val roleId: String = super.create(name)
    expireCache(GLOBAL_SECURITY_ALIAS)
    roleId
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def update(roles: Role*): Unit = {
    super.update(roles: _*)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def update(onConfigurationItem: String, roles: Role*): Unit = {
    super.update(roles: _*)
    expireCache(onConfigurationItem)
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def deleteByName(name: String): Unit = {
    super.deleteByName(name)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def deleteById(roleId: String): Unit = {
    super.deleteById(roleId)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def delete(roleIds: String*): Unit = {
    super.delete(roleIds: _*)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def deleteRoles(onConfigurationItem: String, roleIds: String*): Unit = {
    super.delete(roleIds: _*)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def createOrUpdateRole(role: Role): String = {
    val roleId = super.createOrUpdateRole(role)
    expireCache(GLOBAL_SECURITY_ALIAS)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def rename(name: String, newName: String): String = {
    val roleId = super.rename(name, newName)
    expireCache(GLOBAL_SECURITY_ALIAS)
    roleId
  }

  private def expireCache(onConfigurationItem: String, allEntries: Boolean = false): Unit = {
    val target = if (isGlobalId(Option(onConfigurationItem))) GLOBAL_SECURITY_ALIAS else onConfigurationItem
    broadcastService.broadcast(if (allEntries) EvictAllEvent() else EvictRolesEvent(target), false)
  }
}

// scalastyle:off number.of.methods
class SqlRoleService(securityDatabase: Database) extends RoleService with Logging {

  import securityDatabase._

  type Q = SQuery[Tables.Roles, Tables.Role, Seq]

  type QP = SQuery[(Tables.Roles, Tables.RolePrincipals), (Tables.Role, RolePrincipal), Seq]

  type QRP = SQuery[(Tables.Roles, Tables.RolePermissions), (Tables.Role, Tables.RolePermission), Seq]

  val profile: JdbcProfile = config.databaseType.profile

  import profile.api._

  override def getRoles(rolePattern: String, paging: Paging, order: Ordering): util.List[Role] =
    getGlobalRoles(rolePattern, paging, order).map(toRole(_)).asJavaMutable()

  override def getRoles(onConfigurationItem: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role] =
    runAwait(Tables.roles.filter(_.isOnConfigurationItem(Option(onConfigurationItem))).result).map(toRole(_)).asJavaMutable()

  override def getRolesFor(principal: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role] =
    queryRoles(Seq(principal), rolePattern, paging, order).asJavaMutable()

  override def getRolesFor(auth: Authentication, rolePattern: String, paging: Paging, order: Ordering): util.List[Role] =
    queryRoles(Permissions.authenticationToPrincipals(auth).asScala, rolePattern, paging, order).asJavaMutable()

  private def queryRoles(principalNames: Iterable[String], rolePattern: String, paging: Paging, order: Ordering): Seq[Role] = {
    val query: QP = Tables.roles.join(Tables.rolePrincipals).on(_.id === _.roleId).filter {
      case (role, rolePrincipal) => role.isGlobal && rolePrincipal.principalName.toLowerCase.inLarge(principalNames.map(_.toLowerCase))
    }

    val filters = List(
      (query: QP) => Option(rolePattern) match {
        case Some(pattern) if pattern.nonEmpty => query.filter(_._1.name.toLowerCase.like(s"%${rolePattern.toLowerCase}%"))
        case _ => query
      },
      (query: QP) => Option(order) match {
        case Some(ord) => if (ord.isAscending) query.sortBy(_._1.name.toLowerCase.asc) else query.sortBy(_._1.name.toLowerCase.desc)
        case _ => query
      },
      (query: QP) => withPaging(query, paging)
    )
    runAwait(
      filters
        .foldLeft(query)((acc, filter) => filter(acc))
        .map(_._1).result
    ).map(toRole(_))
  }

  override def getRolesForPermission(permission: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role] = {
    val query: QRP = Tables.roles.join(Tables.rolePermissions)
      .on(_.id === _.roleId)
      .filter(_._1.isGlobal)
      .filter(_._2.permissionName === permission)

    val filters = List(
      (query: QRP) => Option(rolePattern) match {
        case Some(pattern) if pattern.nonEmpty => query.filter(_._1.name.toLowerCase.like(s"%${rolePattern.toLowerCase}%"))
        case _ => query
      },
      (query: QRP) => Option(order) match {
        case Some(ord) => if (ord.isAscending) query.sortBy(_._1.name.toLowerCase.asc) else query.sortBy(_._1.name.toLowerCase.desc)
        case _ => query
      },
      (query: QRP) => withPaging(query, paging)
    )
    runAwait(
      filters
        .foldLeft(query)((acc, filter) => filter(acc))
        .map(_._1).result
    ).map(toRole(_)).asJavaMutable()
  }

  override def getRoleForRoleName(roleName: String): Role = queryRoleAssignments((role, _, _) => role.name === roleName).toList match {
    case role :: _ => role
    case Nil => throw new NotFoundException("Could not find the role [%s]", roleName)
  }

  override def readRoleAssignments(rolePattern: String, paging: Paging, order: Ordering): util.List[Role] =
    queryRoleAssignments(None, rolePattern, paging, order).asJavaMutable()

  override def readRoleAssignments(onConfigurationItem: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role] =
    queryRoleAssignments(Some(onConfigurationItem), rolePattern, paging, order).asJavaMutable()

  private def queryRoleAssignments(onConfigurationItem: Option[String], rolePattern: String, paging: Paging, order: Ordering): Seq[Role] = {
    logger.debug(s"Reading role assignments from [$onConfigurationItem")

    val roleAssignments = queryRoleAssignments((role, _, _) => role.isOnConfigurationItem(onConfigurationItem))

    logger.debug(s"Read from [$onConfigurationItem] role assignments ${roleAssignments.map(formatAssignments)}")
    roleAssignments
  }

  private def queryRoleAssignments(filter: (Tables.Roles, Rep[Option[Tables.RolePrincipals]], Rep[Option[Tables.RoleRoles]]) => Rep[Boolean]): Seq[Role] = {
    val results = runAwait {
      Tables.roles
        .joinLeft(Tables.rolePrincipals).on { case (role, rolePrincipal) => role.id === rolePrincipal.roleId }
        .joinLeft(Tables.roleRoles).on { case ((role, _), roleRole) => role.id === roleRole.roleId }
        .filter { case ((role, rolePrincipal), roleRole) => filter(role, rolePrincipal, roleRole) }
        .map { case ((role, rolePrincipal), roleRole) => (role, rolePrincipal, roleRole) }
        .result
    }

    val memberRoles = results.flatMap(_._3.map(_.memberRoleId)).distinct match {
      case Seq() => Seq.empty
      case roleIds => runAwait(Tables.roles.filter(_.id inLarge roleIds).result)
    }

    results.groupBy(_._1)
      .map { case (role, children) =>
        val roleNames = children.flatMap(_._3).distinct.flatMap(roleRole => memberRoles.find(_.id == roleRole.memberRoleId)).map(_.name)
        val principalNames = children.flatMap(_._2).distinct.map(_.principalName)
        new Role(role.id, role.name, principalNames.asJavaMutable(), roleNames.asJavaMutable())
      }.toList
  }

  override def create(name: String, onConfigurationItem: String): String = {
    create(onConfigurationItem, new Role(name))
    getRoleForRoleName(name).getId
  }

  override def writeRoleAssignments(roles: util.List[Role]): Unit = doWriteRoleAssignments(None, roles.asScala.toSeq)

  override def writeRoleAssignments(onConfigurationItem: String, roles: util.List[Role]): Unit = {
    doWriteRoleAssignments(Some(onConfigurationItem), roles.asScala.toSeq)
  }

  override def createOrUpdateRole(role: Role, onConfigurationItem: String): String = {
    val findFirstGlobalRoleIdByNameOrId = Tables.roles
      .filter(roleRow => roleRow.isGlobal && (roleRow.name === role.getName || roleRow.id === role.getId))
      .map(roleRow => roleRow.id)
      .result

    def createOrUpdateAction(existingIds: Seq[String]) = existingIds match {
      case found if found.exists(_ != role.getId) =>
        failed(new IllegalArgumentException(s"A role named '${role.getName}' already exists."))
      case found if found.contains(role.getId) =>
        updateActions(Seq(role))
      case _ =>
        createActions(Seq(role), Option(onConfigurationItem))

    }

    implicit val ec: ExecutionContext = securityDatabase.database.ioExecutionContext
    runAwaitTry {
      val transaction = for {
        maybeIdAndName <- findFirstGlobalRoleIdByNameOrId
        _ <- createOrUpdateAction(maybeIdAndName)
      } yield role.getId
      transaction.transactionally
    }.get
  }

  override def rename(name: String, newName: String, onConfigurationItem: String): String = {
    val roleId: String = getRoleForRoleName(name).getId
    Tables.roles.filter(_.id === roleId).map(_.name).update(newName)
    roleId
  }

  override def deleteByName(name: String): Unit = {
    delete(getRoleForRoleName(name).getId)
  }

  override def deleteById(roleId: String): Unit = {
    delete(roleId)
  }

  private def doWriteRoleAssignments(onConfigurationItem: Option[String], updatedRoles: Seq[Role]): Unit = {
    logger.debug(s"Writing role assignments ${updatedRoles.map(formatAssignments)} to CI [$onConfigurationItem]")

    val duplicateRoles = updatedRoles.groupBy(_.getName.trim.toLowerCase).filter(_._2.size > 1)
    checkArgument(duplicateRoles.isEmpty, s"Roles with duplicate names [${duplicateRoles.keys.mkString(", ")}] are not allowed")

    updatedRoles.foreach(generateIdIfNecessary)

    val originalRoles = queryRoleAssignments(onConfigurationItem, null, null, null)
    val rolesDiff = Diff(originalRoles.toSet, updatedRoles.toSet)
    lazy val globalRoles = Some(getGlobalRoles(null, null, null))

    runAwait(
      seq(
        deleteActions(rolesDiff.deletedEntries.map(_.getId)),
        createActions(rolesDiff.newEntries, onConfigurationItem, globalRoles),
        updateRoleDiffActions(rolesDiff.updatedEntries)
      ).transactionally
    )
  }

  def create(roles: Role*): Unit = create(null, roles: _*)

  def create(onConfigurationItem: String, roles: Role*): Unit = {
    roles.foreach(generateIdIfNecessary)
    runAwait(createActions(roles, Option(onConfigurationItem)).transactionally)
  }

  private def createActions(roles: Iterable[Role],
                            onConfigurationItem: Option[String],
                            globalRoles: => Option[Iterable[Tables.Role]] = None): DBIOAction[_, NoStream, Effect.Write] = {
    val dbRoles = toDbRoles(roles, onConfigurationItem)
    val dbPrincipals = toDbPrincipals(roles)
    val dbRoleRoles = toDbRoleRoles(roles, globalRoles)

    seq(
      ifNotEmpty(dbRoles)(items => Seq(Tables.roles ++= items)) ++
        ifNotEmpty(dbPrincipals)(items => Seq(Tables.rolePrincipals ++= items)) ++
        ifNotEmpty(dbRoleRoles)(items => Seq(Tables.roleRoles ++= items)): _*
    )
  }

  def update(onConfigurationItem: String, roles: Role*): Unit = this.update(roles: _*)

  def update(roles: Role*): Unit = runAwait(updateActions(roles).transactionally)

  private def updateActions(roles: Iterable[Role]): DBIOAction[_, NoStream, Effect.Write] = updateActions(roles, roles, roles)

  private def updateRoleDiffActions(rolesDiff: Iterable[(Role, Role)]): DBIOAction[_, NoStream, Effect.Write] = {
    val rolesToUpdate = rolesDiff.filter { case (original, updated) => original.getName != updated.getName }.map(_._2)

    val rolesWithPrincipalsToUpdate = rolesDiff.filter { case (original, updated) =>
      val diff = Diff(original.getPrincipals.asScala.toSet, updated.getPrincipals.asScala.toSet)
      diff.newEntries.nonEmpty || diff.deletedEntries.nonEmpty
    }.map(_._2)

    val rolesWithRolesToUpdate = rolesDiff.filter { case (original, updated) =>
      val diff = Diff(original.getRoles.asScala.toSet, updated.getRoles.asScala.toSet)
      diff.newEntries.nonEmpty || diff.deletedEntries.nonEmpty
    }.map(_._2)

    updateActions(rolesToUpdate, rolesWithPrincipalsToUpdate, rolesWithRolesToUpdate)
  }

  private def updateActions(rolesToUpdate: Iterable[Role],
                            rolesWithPrincipalsToUpdate: Iterable[Role],
                            rolesWithRolesToUpdate: Iterable[Role],
                            globalRoles: Option[Iterable[Tables.Role]] = None): DBIOAction[_, NoStream, Effect.Write] = {
    seq(
      ifNotEmpty(rolesToUpdate)(_.map(r => Tables.roles.filter(_.id === r.getId).map(_.name).update(r.getName)).toSeq) ++

        ifNotEmpty(rolesWithPrincipalsToUpdate)(items => Seq(Tables.rolePrincipals.filter(_.roleId inLarge items.map(_.getId)).delete)) ++
        ifNotEmpty(rolesWithPrincipalsToUpdate)(items => Seq(Tables.rolePrincipals ++= toDbPrincipals(items))) ++

        ifNotEmpty(rolesWithRolesToUpdate)(roles => Seq(Tables.roleRoles.filter(_.roleId inLarge roles.map(_.getId)).delete)) ++
        ifNotEmpty(rolesWithRolesToUpdate)(items => Seq(Tables.roleRoles ++= toDbRoleRoles(items, globalRoles))): _*
    )
  }

  def deleteRoles(onConfigurationItem: String, roleIds: String*): Unit = this.delete(roleIds: _*)

  def delete(roleIds: String*): Unit = runAwait(deleteActions(roleIds).transactionally)

  private def deleteActions(roleIds: Iterable[String]): DBIOAction[Unit, NoStream, Effect.Write] = seq(
    ifNotEmpty(roleIds)(roleIds => Seq(Tables.rolePermissions.filter(_.roleId inLarge roleIds).delete)) ++
      ifNotEmpty(roleIds)(roleIds => Seq(Tables.rolePrincipals.filter(_.roleId inLarge roleIds).delete)) ++
      ifNotEmpty(roleIds)(roleIds => Seq(Tables.roleRoles.filter(rr => rr.memberRoleId.inLarge(roleIds) || rr.roleId.inLarge(roleIds)).delete)) ++
      ifNotEmpty(roleIds)(roleIds => Seq(Tables.roles.filter(_.id inLarge  roleIds).delete)): _*
  )

  private def generateIdIfNecessary(role: Role): Unit = {
    if (isEmpty(role.getId) || role.getId == "-1") {
      role.setId(Ids.generate())
    }
  }

  private def toDbRoleRoles(roles: Iterable[Role], globalRolesOpt: => Option[Iterable[Tables.Role]] = None) = {
    lazy val globalRoles = globalRolesOpt.getOrElse(if (roles.isEmpty) Seq.empty else getGlobalRoles(null, null, null))

    roles.flatMap(r => r.getRoles.asScala.map(roleName => Tables.RoleRole(r.getId, roleNameToId(roleName, globalRoles))))
  }

  private def withRolePattern(query: Q, rolePattern: String) = Option(rolePattern) match {
    case Some(pattern) if pattern.nonEmpty => query.filter(_.name.toLowerCase.like(s"%${rolePattern.toLowerCase}%"))
    case _ => query
  }

  private def withOrder(query: Q, order: Ordering) = Option(order) match {
    case Some(ord) => if (ord.isAscending) query.sortBy(_.name.toLowerCase.asc) else query.sortBy(_.name.toLowerCase.desc)
    case _ => query
  }

  private def withPaging[A, B](query: SQuery[A, B, Seq], paging: Paging) = Option(paging) match {
    case Some(p) => query
      .drop((p.page - 1) * p.resultsPerPage)
      .take(p.resultsPerPage)
    case _ => query
  }

  private def getGlobalRoles(rolePattern: String, paging: Paging, order: Ordering): Seq[Tables.Role] = {
    var query: Q = Tables.roles.filter(_.isGlobal)
    query = withRolePattern(query, rolePattern)
    query = withOrder(query, order)
    query = withPaging(query, paging)
    runAwait(query.result)
  }

  private def toDbRoles(roles: Iterable[Role], onConfigurationItem: Option[String]) =
    roles.map(r => Tables.Role(r.getId, r.getName, toDbId(onConfigurationItem)))

  private def toDbPrincipals(roles: Iterable[Role]) =
    roles.flatMap(r => r.getPrincipals.asScala.map(Tables.RolePrincipal(r.getId, _)))

  private def roleNameToId(roleName: String, roles: Iterable[Tables.Role]): String = roles.find(_.name == roleName) match {
    case Some(role) => role.id
    case None => throw new NotFoundException("Role [%s] not found", roleName)
  }

  private def toRole(role: Tables.Role, principals: Seq[Tables.RolePrincipal] = Seq.empty): Role =
    new Role(role.id, role.name, principals.map(_.principalName).asJava)

  private def formatAssignments(role: Role) = s"${role.getName} -> PRINCIPALS=${role.getPrincipals}, ROLES=${role.getRoles}"

  private def ifNotEmpty[T, R](items: Iterable[T])(mapper: Iterable[T] => Seq[R]): Seq[R] = if (items.isEmpty) Seq.empty else mapper(items)

  def countRoles(onConfigurationItem: Number, rolePattern: String): Long = {
    var query = Tables.roles.filter(_.isGlobal).filter(_.ciId === onConfigurationItem.intValue())
    runAwait(withRolePattern(query, rolePattern).length.result).toLong
  }

  def roleExists(roleName: String): Boolean = {
    var query = Tables.roles.filter(_.name === roleName).exists
    runAwait(query.result)
  }

  /**
    * Returns the total of roles for specified CI path
    */
  def countRoles(onConfigurationItem: String, rolePattern: String): Long = {
    var query = Tables.roles.filter(_.isGlobal).filter(_.isOnConfigurationItem(Option(onConfigurationItem)))
    runAwait(withRolePattern(query, rolePattern).length.result).toLong
  }

  override def isReadOnlyAdmin(): Boolean = false

  override def getRoleForRoleId(roleId: String): Role = queryRoleAssignments((role, _, _) => role.id === roleId).toList match {
    case role :: _ => role
    case Nil => throw new NotFoundException(s"Could not find the role with id [$roleId]")
  }
}
// scalastyle:on number.of.methods

case class Diff[T](original: Set[T], updated: Set[T]) {
  lazy val newEntries: Set[T] = updated -- original
  lazy val updatedEntries: Set[(T, T)] = for {
    u <- updated
    o <- original
    if u == o
  } yield (o, u)
  lazy val deletedEntries: Set[T] = original -- updated
}
