package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.status.PhaseStatus
import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.domain.{Phase, Release}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{DependencyPersistence, ReleasePersistence, TaskPersistence}
import com.xebialabs.xlrelease.repository.{CommentRepository, DependencyTargetLoader, FacetRepositoryDispatcher, PhaseRepository}

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlPhaseRepository(releasePersistence: ReleasePersistence,
                         taskPersistence: TaskPersistence,
                         val configurationPersistence: ConfigurationReferencePersistence,
                         val commentRepository: CommentRepository,
                         val dependencyTargetLoader: DependencyTargetLoader,
                         val dependencyPersistence: DependencyPersistence,
                         val facetRepositoryDispatcher: FacetRepositoryDispatcher,
                         val repositoryAdapter: SqlRepositoryAdapter)
  extends PhaseRepository
    with BaseReleaseItemRepository
    with ConfigurationReferencesSupport {

  @Timed
  override def findById(phaseId: String): Phase = {
    val release = getRelease(releaseIdFrom(phaseId), phaseId)
    notNull(release.getPhase(phaseId.normalized), phaseId)
  }

  @Timed
  override def create(release: Release, phase: Phase): Phase = {
    releasePersistence.update(release)
    phase.getAllTasks.asScala.foreach { task =>
      val taskUid = taskPersistence.insert(task, release.getCiUid)
      task.dependencies.foreach(insertDependencyWithGateTaskUid(taskUid))
    }
    updateConfigurationRefs(release)
    facetRepositoryDispatcher.liveRepository.createFromTasks(phase.getAllTasks.asScala.toSeq)
    phase
  }

  @Timed
  override def delete(release: Release, phase: Phase): Phase = {
    checkIsNotReferencedByDependencies(phase.getId)
    releasePersistence.update(release)
    phase.getAllTasks.asScala.foreach { task =>
      task.dependencies.foreach(dependencyPersistence.deleteDependency)
      commentRepository.deleteByTask(task)
      taskPersistence.delete(task)
    }
    updateConfigurationRefs(release)
    phase
  }

  @Timed
  override def update(release: Release, original: Phase, updated: Phase): Phase = {
    releasePersistence.update(release)
    // phase update never changes underlying tasks or dependencies
    updated
  }

  @Timed
  override def move(release: Release, movedPhase: Phase): Phase = {
    releasePersistence.update(release)
    movedPhase
  }

  @Timed
  @IsReadOnly
  override def getTitle(phaseId: String): String = {
    val target = dependencyTargetLoader.getDependencyNodeData(phaseId)
    target.phaseTitle
  }

  override def getStatus(phaseId: String): PhaseStatus = {
    val target = dependencyTargetLoader.getDependencyNodeData(phaseId)
    target.phaseStatus
  }
}
