package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.UserToken
import com.xebialabs.xlrelease.repository.UserTokenRepository
import com.xebialabs.xlrelease.repository.UserTokenRepository._
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.UserTokenQueryBuilder.STMT_USER_TOKEN_SELECT
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_TOKENS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport, Utils}
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.util.Date

@IsTransactional
class SqlUserTokenRepository(implicit val dialect: Dialect,
                             implicit val jdbcTemplate: JdbcTemplate,
                             implicit val sqlRepositoryAdapter: SqlRepositoryAdapter)
  extends UserTokenRepository
    with PersistenceSupport
    with Utils
    with UserTokenMapper
    with Logging {

  private val STMT_INSERT_USER_TOKEN =
    s"""INSERT INTO ${USER_TOKENS.TABLE} (
       |  ${USER_TOKENS.USERNAME},
       |  ${USER_TOKENS.TOKEN_NOTE},
       |  ${USER_TOKENS.TOKEN_HASH},
       |  ${USER_TOKENS.CREATED_DATE},
       |  ${USER_TOKENS.EXPIRY_DATE}
       | ) VALUES (
       |  :${USER_TOKENS.USERNAME},
       |  :${USER_TOKENS.TOKEN_NOTE},
       |  :${USER_TOKENS.TOKEN_HASH},
       |  :${USER_TOKENS.CREATED_DATE},
       |  :${USER_TOKENS.EXPIRY_DATE}
       | )
       |""".stripMargin

  @Timed
  override def create(userToken: UserToken, tokenHash: String): CiUid = {
    logger.trace(s"creating user token $userToken")
    val params = Map(
      USER_TOKENS.USERNAME -> userToken.username.toLowerCase(),
      USER_TOKENS.TOKEN_NOTE -> userToken.tokenNote,
      USER_TOKENS.TOKEN_HASH -> tokenHash,
      USER_TOKENS.CREATED_DATE -> userToken.createdDate,
      USER_TOKENS.EXPIRY_DATE -> userToken.expiryDate
    )

    sqlInsert(pkName(USER_TOKENS.CI_UID), STMT_INSERT_USER_TOKEN, params)
  }

  private val STMT_UPDATE_LAST_USED =
    s"""UPDATE ${USER_TOKENS.TABLE}
       |  SET ${USER_TOKENS.LAST_USED_DATE} = CASE
       |    WHEN ${USER_TOKENS.LAST_USED_DATE} < :lastUsed OR ${USER_TOKENS.LAST_USED_DATE} IS NULL THEN :lastUsed
       |    ELSE ${USER_TOKENS.LAST_USED_DATE}
       |  END
       |  WHERE ${USER_TOKENS.CI_UID} = :ciUid
       |""".stripMargin

  @Timed
  override def updateLastUsedBatch(entries: Map[Integer, Date]): Int = {
    sqlBatch(STMT_UPDATE_LAST_USED, entries.map {
      case (ciUid, lastUsed) =>
        params(
          "ciUid" -> ciUid,
          "lastUsed" -> lastUsed
        )
    }.toSet)
  }.sum

  private val STMT_UPDATE_TOKEN_EXPIRE_NOTIFIED =
    s"""UPDATE ${USER_TOKENS.TABLE}
       |  SET ${USER_TOKENS.IS_TOKEN_EXPIRE_NOTIFIED} = 1
       |  WHERE ${USER_TOKENS.CI_UID} = :ciUid
       |""".stripMargin

  @Timed
  override def updateTokenExpiredNotified(ciUid: CiUid): Unit = {
    sqlUpdate(STMT_UPDATE_TOKEN_EXPIRE_NOTIFIED, params("ciUid" -> ciUid), _ => ())
  }

  private val STMT_DELETE_USER_TOKEN_BY_CIUID = s"DELETE FROM ${USER_TOKENS.TABLE} WHERE ${USER_TOKENS.CI_UID} = :${USER_TOKENS.CI_UID}"

  @Timed
  override def delete(ciUid: CiUid): Unit = {
    logger.trace(s"deleting user token with ciUid[$ciUid]")
    sqlExec(STMT_DELETE_USER_TOKEN_BY_CIUID, params(USER_TOKENS.CI_UID -> ciUid), _.execute())
  }

  private val STMT_DELETE_USER_TOKEN_BY_USERNAME = s"DELETE FROM ${USER_TOKENS.TABLE} WHERE ${USER_TOKENS.USERNAME} = :${USER_TOKENS.USERNAME}"

  @Timed
  override def delete(username: String): Unit = {
    logger.trace(s"deleting all user tokens with username[$username]")
    sqlExec(STMT_DELETE_USER_TOKEN_BY_USERNAME, params(USER_TOKENS.USERNAME -> username.toLowerCase), _.execute())
  }

  private val QUERY_USER_TOKEN_BY_CIUID =
    s"""
       |$STMT_USER_TOKEN_SELECT
       |  WHERE ${USER_TOKENS.CI_UID} = ?
       |""".stripMargin

  @Timed
  @IsReadOnly
  override def findByCiUid(ciUid: CiUid): Option[UserToken] = {
    logger.trace(s"finding user token with ciUid[$ciUid]")
    findOptional(_.queryForObject(QUERY_USER_TOKEN_BY_CIUID, userTokenMapper, ciUid))
  }

  private val QUERY_USER_TOKEN_BY_USER_AND_NOTE =
    s"""
       |$STMT_USER_TOKEN_SELECT
       |  WHERE ${USER_TOKENS.USERNAME} = ?
       |  AND ${USER_TOKENS.TOKEN_NOTE} = ?
       |""".stripMargin

  @Timed
  @IsReadOnly
  override def findByUserAndNote(username: String, tokenNote: String): Option[UserToken] = {
    logger.trace(s"finding user token with username[$username] and tokenNote[$tokenNote]")
    findOptional(_.queryForObject(QUERY_USER_TOKEN_BY_USER_AND_NOTE, userTokenMapper, username.toLowerCase, tokenNote))
  }

  private val QUERY_USER_TOKEN_BY_TOKEN_HASH =
    s"""
       |$STMT_USER_TOKEN_SELECT
       |  WHERE ${USER_TOKENS.TOKEN_HASH} = ?
       |""".stripMargin

  @Timed
  @IsReadOnly
  override def findByUserToken(tokenHash: String): Option[UserToken] = {
    findOptional(_.queryForObject(QUERY_USER_TOKEN_BY_TOKEN_HASH, userTokenMapper, tokenHash))
  }

  @Timed
  @IsReadOnly
  override def query(operation: QueryOperation, pageable: Pageable): Page[UserToken] = {
    UserTokenQueryBuilder(dialect, namedTemplate, sqlRepositoryAdapter)
      .from(operation)
      .withPageable(pageable)
      .build()
      .execute()
  }

}

trait UserTokenMapper {
  def sqlRepositoryAdapter: SqlRepositoryAdapter

  private[repository] def userTokenMapper: RowMapper[UserToken] = (rs, _) => {
    val userToken: UserToken = new UserToken
    userToken.setCiUid(rs.getInt(USER_TOKENS.CI_UID))
    userToken.setUsername(rs.getString(USER_TOKENS.USERNAME))
    userToken.setTokenNote(rs.getString(USER_TOKENS.TOKEN_NOTE))
    userToken.setCreatedDate(rs.getTimestamp(USER_TOKENS.CREATED_DATE))
    userToken.setLastUsedDate(rs.getTimestamp(USER_TOKENS.LAST_USED_DATE))
    userToken.setExpiryDate(rs.getTimestamp(USER_TOKENS.EXPIRY_DATE))
    userToken.setExpiryNotified(rs.getInt(USER_TOKENS.IS_TOKEN_EXPIRE_NOTIFIED).asBoolean)
    userToken
  }
}

trait UserTokenQueryBuilder {
  def from(operation: QueryOperation): FiltersQueryBuilder[QueryOperation, UserToken]
}

object UserTokenQueryBuilder {
  val STMT_USER_TOKEN_SELECT: String =
    s"""
       |SELECT
       |  ${USER_TOKENS.CI_UID},
       |  ${USER_TOKENS.USERNAME},
       |  ${USER_TOKENS.TOKEN_NOTE},
       |  ${USER_TOKENS.CREATED_DATE},
       |  ${USER_TOKENS.LAST_USED_DATE},
       |  ${USER_TOKENS.EXPIRY_DATE},
       |  ${USER_TOKENS.IS_TOKEN_EXPIRE_NOTIFIED}
       | FROM ${USER_TOKENS.TABLE}
       |""".stripMargin

  def apply(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate, sqlRepositoryAdapter: SqlRepositoryAdapter) =
    new SqlUserTokenQueryBuilder(dialect, namedTemplate, sqlRepositoryAdapter)
}

class SqlUserTokenQueryBuilder(dialect: Dialect,
                               namedTemplate: NamedParameterJdbcTemplate,
                               sqlRepositoryAdapter: SqlRepositoryAdapter)
  extends UserTokenQueryBuilder {

  override def from(operation: QueryOperation): FiltersQueryBuilder[QueryOperation, UserToken] =
    new SqlUserTokenFilterQueryBuilder(dialect, namedTemplate, sqlRepositoryAdapter).from(operation)

}

class SqlUserTokenFilterQueryBuilder(val dialect: Dialect,
                                     val namedTemplate: NamedParameterJdbcTemplate,
                                     val sqlRepositoryAdapter: SqlRepositoryAdapter)
  extends FiltersQueryBuilder[QueryOperation, UserToken]
    with FilterQueryBuilderSupport[QueryOperation, UserToken]
    with UserTokenMapper {

  private lazy val queryTemplate =
    s"""
       |$STMT_USER_TOKEN_SELECT
       | $whereClause
       | $orderClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       |  FROM ${USER_TOKENS.TABLE}
       |  $whereClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  override def from(operation: QueryOperation): SelfType = {
    operation match {
      case ByUsername(username) =>
        if (isNotBlank(username)) {
          equal(USER_TOKENS.USERNAME, "username", username.toLowerCase())
        }
      case TokensAboutToExpire(expiryDate) =>
        this.whereClauses +=
          s"""
             |(${USER_TOKENS.EXPIRY_DATE} IS NOT NULL
             |  AND ${USER_TOKENS.EXPIRY_DATE} > :currentTimestamp
             |  AND ${USER_TOKENS.EXPIRY_DATE} <= :expiryDate)
             |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")
        this.whereClauses += s"${USER_TOKENS.IS_TOKEN_EXPIRE_NOTIFIED} = 0"
        this.queryParams += "expiryDate" -> expiryDate.asTimestamp
        this.queryParams += "currentTimestamp" -> new Date().asTimestamp
    }
    this
  }

  override def build(): PageableQuery[UserToken] = {
    remapSortOrderParams()
    buildSortOrderClause("ciUid")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[UserToken](namedTemplate, resultsQueryString, queryParams.toMap, userTokenMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[UserToken](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      "ciUid" -> USER_TOKENS.CI_UID
    )
  }
}
