package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, ReleasePersistence}
import com.xebialabs.xlrelease.repository.{TemplateMetadata, TemplateMetadataRepository}
import grizzled.slf4j.Logging

@IsTransactional
class SqlTemplateMetadataRepository(releasePersistence: ReleasePersistence) extends TemplateMetadataRepository with Logging {

  @Timed
  override def createOrUpdate(templateMeta: TemplateMetadata): TemplateMetadata = {
    val releaseUid = templateMeta.releaseUid
    releasePersistence.findTemplateMetadata(releaseUid) match {
      case Some(v) =>
        releasePersistence.updateTemplateMetadata(releaseUid, templateMeta)
        if (templateMeta.templateLogo != null) {
          releasePersistence.updateLogo(releaseUid, Option(v.templateLogo), templateMeta.templateLogo)
        }
      case None =>
        releasePersistence.createTemplateMetadata(templateMeta)
    }
    templateMeta
  }

  @Timed
  override def update(releaseUid: CiUid, templateMetadata: TemplateMetadata): TemplateMetadata = {
    releasePersistence.updateTemplateMetadata(releaseUid, templateMetadata)
    templateMetadata
  }

  @Timed
  override def createOrUpdateLogo(releaseUid: CiUid, templateLogo: TemplateLogo): TemplateLogo = {
    releasePersistence.findTemplateMetadata(releaseUid) match {
      case Some(v) =>
        releasePersistence.updateLogo(releaseUid, Option(v.templateLogo), templateLogo)
      case None =>
        // create new template metadata
        val templateMetadata = TemplateMetadata(releaseUid, null, null, null, allowTargetFolderOverride = true, templateLogo)
        releasePersistence.createTemplateMetadata(templateMetadata)
    }
    releasePersistence.findTemplateMetadata(releaseUid).get.templateLogo
  }

  @Timed
  override def delete(releaseUid: CiUid): Unit = {
    releasePersistence.deleteTemplateMetadata(releaseUid)
  }

  @Timed
  @IsReadOnly
  override def find(releaseUid: CiUid): Option[TemplateMetadata] = {
    releasePersistence.findTemplateMetadata(releaseUid)
  }

  @Timed
  @IsReadOnly
  override def findTemplateLogoById(logoId: String): Option[TemplateLogo] = {
    logger.debug(s"Searching for template logo [$logoId]")
    releasePersistence.findTemplateLogoById(logoId)
  }
}
