package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.repository.sql.persistence.CompressionSupport.BUFFER_SIZE
import org.apache.commons.compress.compressors.CompressorStreamFactory
import org.apache.commons.compress.compressors.deflate.{DeflateCompressorOutputStream, DeflateParameters}
import org.apache.commons.compress.utils.IOUtils

import java.io.{ByteArrayInputStream, ByteArrayOutputStream, InputStream}
import java.nio.charset.StandardCharsets.UTF_8
import java.util.zip.Deflater
import scala.util.Using

object CompressionSupport {
  private val BUFFER_SIZE = 4096
}

trait CompressionSupport {

  /**
    * Compresses the text using Deflate compression algorithm.
    *
    * @param text the input text, e.g. a release JSON.
    * @return compressed text as array of bytes.
    */
  def compress(text: String): Array[Byte] = {
    val input = text.getBytes(UTF_8)
    compress(input)
  }

  /**
    * Decompresses the input stream and converts to a string using UTF-8 encoding.
    *
    * @param inputStream the input stream, NOT closed by this method.
    * @return the same string as was passed before to the [[compress(String)}]] method.
    */
  def decompress(inputStream: InputStream): String = {
    val byteArray = decompressInputStream(inputStream)
    new String(byteArray, UTF_8)
  }

  def compress(byteArray: Array[Byte]): Array[Byte] = {
    val inputBytesStream = new ByteArrayInputStream(byteArray)
    val compressedBytesStream = new ByteArrayOutputStream(byteArray.length)

    val parameters = new DeflateParameters()
    parameters.setCompressionLevel(Deflater.BEST_SPEED)

    Using.resource(new DeflateCompressorOutputStream(compressedBytesStream, parameters)) { compressorOutputStream =>
      IOUtils.copy(inputBytesStream, compressorOutputStream, BUFFER_SIZE)
    }

    compressedBytesStream.toByteArray
  }

  def decompress(byteArray: Array[Byte]): Array[Byte] = {
    val inputStream = new ByteArrayInputStream(byteArray)
    decompressInputStream(inputStream)
  }

  private def decompressInputStream(inputStream: InputStream): Array[Byte] = {
    val decompressorInputStream = new CompressorStreamFactory()
      .createCompressorInputStream(CompressorStreamFactory.DEFLATE, inputStream)
    val decompressedBytesStream = new ByteArrayOutputStream(BUFFER_SIZE)

    IOUtils.copy(decompressorInputStream, decompressedBytesStream)

    decompressedBytesStream.toByteArray
  }

}
