package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.xlrelease.api.v1.filter.CategoryFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.Category
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.CategoryMapper
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CATEGORIES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.RichJBooleanAsInt
import com.xebialabs.xlrelease.repository.sql.query.CategoryQueryBuilder.STMT_CATEGORY_SELECT
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.util.StringUtils

trait CategoryQueryBuilder {
  def from(filter: CategoryFilters): FiltersQueryBuilder[CategoryFilters, Category]
}

object CategoryQueryBuilder {
  val STMT_CATEGORY_SELECT: String =
    s"""
       |SELECT
       |  ${CATEGORIES.CI_UID},
       |  ${CATEGORIES.CATEGORY},
       |  ${CATEGORIES.ACTIVE}
       | FROM ${CATEGORIES.TABLE}
       |""".stripMargin

  def apply(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate) =
    new SqlCategoryQueryBuilder(dialect, namedTemplate)
}

class SqlCategoryQueryBuilder(dialect: Dialect,
                              namedTemplate: NamedParameterJdbcTemplate)
  extends CategoryQueryBuilder {

  override def from(filter: CategoryFilters): FiltersQueryBuilder[CategoryFilters, Category] =
    new SqlCategoryFilterQueryBuilder(dialect, namedTemplate).from(filter)

}

class SqlCategoryFilterQueryBuilder(val dialect: Dialect,
                                    val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[CategoryFilters, Category]
    with FilterQueryBuilderSupport[CategoryFilters, Category]
    with CategoryMapper {

  private lazy val queryTemplate =
    s"""
       |$STMT_CATEGORY_SELECT
       | $whereClause
       | $orderClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       |  FROM ${CATEGORIES.TABLE}
       |  $whereClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  override def from(filter: CategoryFilters): SelfType = {
    if (StringUtils.hasText(filter.getCategory)) {
      like(s"${CATEGORIES.CATEGORY}", CATEGORIES.CATEGORY, filter.getCategory)
    }
    if (null != filter.getActive) {
      whereClauses += s"${CATEGORIES.ACTIVE} = :${CATEGORIES.ACTIVE}"
      queryParams += CATEGORIES.ACTIVE -> filter.getActive.asInteger
    }
    this
  }

  override def build(): PageableQuery[Category] = {
    remapSortOrderParams()
    buildSortOrderClause(CATEGORIES.CATEGORY)
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[Category](namedTemplate, resultsQueryString, queryParams.toMap, categoryMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[Category](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      CATEGORIES.CATEGORY -> s"${CATEGORIES.CATEGORY}"
    )
  }
}
