package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.repository.{FolderRepository, PersistenceInterceptor, ReleaseRepository}
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

@Component
@IsTransactional
class ReleasePersistenceInterceptor(releaseRepository: ReleaseRepository,
                                    folderRepository: FolderRepository)
  extends PersistenceInterceptor[Release] {

  releaseRepository.registerPersistenceInterceptor(this)

  private def validateDefaultTargetFolder(release: Release): Unit = {
    val doesNotAllowTargetFolderOverride = !release.getAllowTargetFolderOverride
    val defaultTargetFolderId = release.getDefaultTargetFolderId
    val defaultTargetFolderIdIsEmpty = !StringUtils.hasText(defaultTargetFolderId)
    if (release.isTemplate && doesNotAllowTargetFolderOverride && defaultTargetFolderIdIsEmpty) {
      val msg = s"Override of the target folder is not allowed, but default target folder is not provided for release with ID [${release.getId}]"
      throw new IncorrectArgumentException(msg)
    }
    if (!defaultTargetFolderIdIsEmpty) {
      val optionalFolder = folderRepository.findById(defaultTargetFolderId, depth = 1)
      if (optionalFolder.isEmpty) {
        throw new IncorrectArgumentException(s"Cannot find folder with id [$defaultTargetFolderId]")
      }
    }
  }

  override def onCreate(ci: Release): Unit = {
    validateDefaultTargetFolder(ci)
  }

  override def onUpdate(ci: Release): Unit = {
    validateDefaultTargetFolder(ci)
  }
}
