package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.xlrelease.api.v1.filter.CategoryFilters
import com.xebialabs.xlrelease.api.v1.views.CategoryView
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.CategoryMapper
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{CATEGORIES, RELEASE_CATEGORY_REFS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.RichJBooleanAsInt
import com.xebialabs.xlrelease.repository.sql.query.CategoryQueryBuilder.{STMT_CATEGORY_WITH_COUNT_SELECT, catAlias}
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.util.StringUtils

trait CategoryQueryBuilder {
  def from(filter: CategoryFilters): FiltersQueryBuilder[CategoryFilters, CategoryView]
}

object CategoryQueryBuilder {
  val catAlias = "cat"
  val refAlias = "rcr"

  val STMT_CATEGORY_WITH_COUNT_SELECT: String =
    s"""
       |SELECT
       |  $catAlias.${CATEGORIES.CI_UID},
       |  $catAlias.${CATEGORIES.CATEGORY},
       |  $catAlias.${CATEGORIES.ACTIVE},
       |  COUNT($refAlias.${RELEASE_CATEGORY_REFS.targetColumn})
       | FROM ${CATEGORIES.TABLE} $catAlias
       | LEFT JOIN ${RELEASE_CATEGORY_REFS.TABLE} $refAlias ON $catAlias.${CATEGORIES.CI_UID} = $refAlias.${RELEASE_CATEGORY_REFS.targetColumn}
       |""".stripMargin

  val STMT_CATEGORY_SELECT: String =
    s"""
       |SELECT
       |  $catAlias.${CATEGORIES.CI_UID},
       |  $catAlias.${CATEGORIES.CATEGORY},
       |  $catAlias.${CATEGORIES.ACTIVE}
       | FROM ${CATEGORIES.TABLE} $catAlias
       |""".stripMargin

  def apply(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate) =
    new SqlCategoryQueryBuilder(dialect, namedTemplate)
}

class SqlCategoryQueryBuilder(dialect: Dialect,
                              namedTemplate: NamedParameterJdbcTemplate)
  extends CategoryQueryBuilder {

  override def from(filter: CategoryFilters): FiltersQueryBuilder[CategoryFilters, CategoryView] =
    new SqlCategoryFilterQueryBuilder(dialect, namedTemplate).from(filter)
}

class SqlCategoryFilterQueryBuilder(val dialect: Dialect,
                                    val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[CategoryFilters, CategoryView]
    with FilterQueryBuilderSupport[CategoryFilters, CategoryView]
    with CategoryMapper {

  private lazy val queryTemplate =
    s"""
       | $STMT_CATEGORY_WITH_COUNT_SELECT
       | $whereClause
       | GROUP BY $catAlias.${CATEGORIES.CI_UID}, $catAlias.${CATEGORIES.CATEGORY}, $catAlias.${CATEGORIES.ACTIVE}
       | $orderClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       |  FROM ${CATEGORIES.TABLE} $catAlias
       |  $whereClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  override def from(filter: CategoryFilters): SelfType = {
    if (StringUtils.hasText(filter.getCategory)) {
      like(s"$catAlias.${CATEGORIES.CATEGORY}", CATEGORIES.CATEGORY, filter.getCategory)
    }
    if (null != filter.getActive) {
      whereClauses += s"$catAlias.${CATEGORIES.ACTIVE} = :${CATEGORIES.ACTIVE}"
      queryParams += CATEGORIES.ACTIVE -> filter.getActive.asInteger
    }
    this
  }

  override def build(): PageableQuery[CategoryView] = {
    remapSortOrderParams()
    buildSortOrderClause(CATEGORIES.CATEGORY)
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[CategoryView](namedTemplate, resultsQueryString, queryParams.toMap, categoryCountMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[CategoryView](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      CATEGORIES.CATEGORY -> s"${CATEGORIES.CATEGORY}"
    )
  }
}
