package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.json.JsonUtils.TaskFacetJsonOps
import com.xebialabs.xlrelease.repository.FacetRepository
import com.xebialabs.xlrelease.repository.FacetRepository.GenericFacetRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FACETS
import com.xebialabs.xlrelease.repository.sql.persistence.data.FacetRow
import com.xebialabs.xlrelease.repository.sql.persistence.{FacetPersistence, FacetSqlBuilder}
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlFacetRepository(val ciIdService: CiIdService,
                         val facetPersistence: FacetPersistence,
                         val sqlRepositoryAdapter: SqlRepositoryAdapter,
                         implicit val sqlDialect: Dialect) extends GenericFacetRepository with FacetRepository.ForLive with Logging with FacetFetcher {

  override def get(facetId: String, facetType: Option[Type] = None): Facet = {
    facetPersistence.get(facetId)
      .map(row => toFacet(row))
      .getOrElse(throw new NotFoundException(s"Facet '$facetId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    facetPersistence.create(facet)
  }

  override def delete(facetId: String, facetType: Option[Type] = None): Unit = facetPersistence.delete(facetId)

  override def update(facet: Facet): Facet = facetPersistence.update(facet)

  override def exists(facetId: String, facetType: Option[Type] = None): Boolean = facetPersistence.exists(facetId)

  override def search(facetsFilters: FacetFilters): Seq[Facet] = {
    val query = new FacetSqlBuilder()
      .select
      .withFacetTypes(Option(facetsFilters.getTypes).map(_.asScala.toSeq).getOrElse(Seq.empty))
      .withParentId(facetsFilters.getParentId)
      .withTargetId(facetsFilters.getTargetId)
      .orderBy(FACETS.CI_UID)
      .build()

    facetPersistence.search(query).map(row => toFacet(row))
  }

  val supportsType: Option[Type] = None
}

private[sql] trait FacetFetcher {

  def facetPersistence: FacetPersistence
  def sqlRepositoryAdapter: SqlRepositoryAdapter

  def findAllFacetsByRelease(release: Release): Seq[Facet] = {
    facetPersistence.findByParentId(release.getId).filter(_.content.isRegistered).map { row =>
      toFacet(row)
    }
  }

  def findAllFacetsByTask(task: Task): Seq[Facet] = {
    facetPersistence.findByTargetId(task.getId).filter(_.content.isRegistered).map { row =>
      toFacet(row)
    }
  }

  protected def toFacet(row: FacetRow): Facet = {
    val facetOption = sqlRepositoryAdapter.deserialize[Facet](row.content.withoutUnknownProps)
    facetOption match {
      case Some(facet) =>
        facet.setTargetId(row.targetId)
        facet
      case None =>
        throw new NotFoundException(s"Error reading facet '${row.ciUid}', see logs for more details.")
    }

  }

}
