package com.xebialabs.xlrelease.security

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.security.Role
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_PROFILE
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.security.SqlReleaseRoleRepository._
import com.xebialabs.xlrelease.security.sql.SqlRoleService
import com.xebialabs.xlrelease.security.sql.db.Ids.GLOBAL_SCOPE_CI_ID
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PRINCIPALS}
import com.xebialabs.xlrelease.views.{RolePrincipalsView, RoleView, UserView}
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, PageImpl, Pageable}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.util.Assert
import org.springframework.util.StringUtils.hasText

import java.util
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlReleaseRoleRepository(val jdbcTemplate: JdbcTemplate,
                               val dialect: Dialect,
                               sqlRoleService: SqlRoleService)
  extends ReleaseRoleRepository with GlobalRolePrincipalTableViewMapper with PersistenceSupport with LimitOffset with Logging {

  @Timed
  @IsReadOnly
  override def getGlobalRolePrincipals(pageable: Pageable): Page[RolePrincipalsView] = {
    Assert.notNull(pageable, "pageable cannot be null")
    val dataQuery = getQueryForGlobalRolePrincipals(pageable)
    val globalRoles = sqlQuery(
      dataQuery,
      params(),
      globalRolePrincipalViewMapper
    )

    val totalCount: Long = findOne(sqlQuery(STMT_GET_TOTAL_GLOBAL_ROLE_COUNT, params(), rs => rs.getLong(1))).get

    val results = globalRoles.groupBy(_.name).map {
      case (roleName, children) =>
        toRolePrincipalsView(children.toSeq)
    }.toList.sorted((o1: RolePrincipalsView, o2: RolePrincipalsView) =>
      o1.getRole.getName.compareTo(o2.getRole.getName)
    ).asJava

    new PageImpl[RolePrincipalsView](results, pageable, totalCount)
  }

  @Timed
  @IsReadOnly
  override def getGlobalRolePrincipal(roleName: String): Option[RolePrincipalsView] = {
    val globalRole = sqlQuery(getQueryForGlobalRolePrincipal(roleName), params(ROLES.name -> roleName.trim.toLowerCase()), globalRolePrincipalViewMapper)
    if (globalRole.isEmpty) {
      None
    } else {
      Option(toRolePrincipalsView(globalRole.toSeq))
    }
  }

  @Timed
  override def create(onConfigurationItem: String, roles: Role*): Unit = {
    // TODO rewrite to use spring
    sqlRoleService.create(onConfigurationItem, roles: _*)
  }

  @Timed
  override def deleteGlobalRole(roleId: String): Unit = {
    sqlRoleService.deleteById(roleId)
  }

  @Timed
  override def globalRoleExists(roleName: String): Boolean = {
    sqlRoleService.globalRoleExists(roleName)
  }

  @Timed
  override def update(onConfigurationItem: String, roles: Role*): Unit = {
    sqlRoleService.update(onConfigurationItem, roles: _*)
  }

  @IsReadOnly
  @Timed
  override def getGlobalRole(roleName: String): Option[Role] = {
    sqlRoleService.getGlobalRole(roleName)
  }

  @IsReadOnly
  @Timed
  override def getGlobalRoles(rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] = {
    sqlRoleService.getRoles(rolePattern, paging, order)
  }

  @Timed
  override def renameGlobalRole(oldName: String, newName: String): Unit = {
    sqlRoleService.rename(oldName, newName)
  }

  private def getQueryForGlobalRolePrincipals(pageable: Pageable): String = {
    Assert.notNull(pageable, "pageable cannot be null")
    val whereClause = if (pageable.isUnpaged) {
      s"WHERE $rolesAlias.${ROLES.ciId} = $GLOBAL_SCOPE_CI_ID ORDER BY $rolesAlias.${ROLES.name}"
    } else {
      val rolesSubQuery = s"SELECT r.${ROLES.id} FROM ${ROLES.TABLE} r WHERE r.${ROLES.ciId} = $GLOBAL_SCOPE_CI_ID ORDER BY r.${ROLES.name}"
      val subQuery = addLimitAndOffset(rolesSubQuery, pageable.getPageSize, pageable.getOffset)
      s"WHERE $rolesAlias.${ROLES.id} IN ($subQuery)"
    }

    val query =
      s"""
         |$STMT_BASE_GET_GLOBAL_ROLE_PRINCIPALS
         | $whereClause
         |""".stripMargin
    query
  }

  private def getQueryForGlobalRolePrincipal(roleName: String): String = {
    Assert.hasText(roleName, "roleName cannot be null")
    val query =
      s"""
         |$STMT_BASE_GET_GLOBAL_ROLE_PRINCIPALS
         | WHERE $rolesAlias.${ROLES.ciId} = $GLOBAL_SCOPE_CI_ID
         |  AND LOWER($rolesAlias.${ROLES.name}) = :${ROLES.name}
         |""".stripMargin
    query
  }

  private def toRolePrincipalsView(globalRoleTableViews: Seq[GlobalRolePrincipalTableView]): RolePrincipalsView = {
    val roleView = new RoleView()
    roleView.setId(globalRoleTableViews.head.id)
    roleView.setName(globalRoleTableViews.head.name)
    val principals = globalRoleTableViews.filter(g => hasText(g.principalName)).map(r => new UserView(r.principalName, r.fullName)).toList.asJava
    val rolePrincipalsView = new RolePrincipalsView()
    rolePrincipalsView.setRole(roleView)
    rolePrincipalsView.setPrincipals(principals)

    rolePrincipalsView
  }
}

object SqlReleaseRoleRepository {
  private val rolesAlias = "roles"
  private val rolePrincipalsAlias = "rolePrincipals"
  private val userProfilesAlias = "userProfiles"

  private val STMT_BASE_GET_GLOBAL_ROLE_PRINCIPALS: String =
    s"""SELECT
       | $rolesAlias.${ROLES.id},
       | $rolesAlias.${ROLES.name},
       | $rolesAlias.${ROLES.ciId},
       | $rolePrincipalsAlias.${ROLE_PRINCIPALS.principalName},
       | $userProfilesAlias.${USER_PROFILE.FULL_NAME}
       | FROM ${ROLES.TABLE} $rolesAlias
       |  LEFT OUTER JOIN ${ROLE_PRINCIPALS.TABLE} $rolePrincipalsAlias ON $rolesAlias.${ROLES.id} = $rolePrincipalsAlias.${ROLE_PRINCIPALS.roleId}
       |  LEFT OUTER JOIN ${USER_PROFILE.TABLE} $userProfilesAlias ON LOWER($rolePrincipalsAlias.${ROLE_PRINCIPALS.principalName}) = $userProfilesAlias.${USER_PROFILE.USERNAME}
       |""".stripMargin

  private val STMT_GET_TOTAL_GLOBAL_ROLE_COUNT: String = s"SELECT COUNT(1) from ${ROLES.TABLE} r WHERE r.${ROLES.ciId} = $GLOBAL_SCOPE_CI_ID"

  private[security] case class GlobalRolePrincipalTableView(id: String, name: String, ciId: Int, principalName: String, fullName: String)
}

trait GlobalRolePrincipalTableViewMapper {
  private[security] def globalRolePrincipalViewMapper: RowMapper[GlobalRolePrincipalTableView] = (rs, _) => {
    GlobalRolePrincipalTableView(
      id = rs.getString(ROLES.id),
      name = rs.getString(ROLES.name),
      ciId = rs.getInt(ROLES.ciId),
      principalName = rs.getString(ROLE_PRINCIPALS.principalName),
      fullName = rs.getString(USER_PROFILE.FULL_NAME)
    )
  }
}
