package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.security.{PermissionEnforcer, Permissions}
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.status.FlagStatus.{ATTENTION_NEEDED, AT_RISK}
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder
import com.xebialabs.xlrelease.security.XLReleasePermissions.{AUDIT_ALL, VIEW_RELEASE, VIEW_WORKFLOW_EXECUTION}
import com.xebialabs.xlrelease.utils.ScopedTokenPermissionValidator.checkPermissionSupportedinScopedToken

import scala.jdk.CollectionConverters._

object SqlReleasesFilterSupport extends SqlFilterSupport[ReleasesFilters, ReleasesSqlBuilder] {

  //noinspection ScalaStyle
  override def sqlBuilderByFilters(filters: ReleasesFilters, principals: Iterable[String], roleIds: Iterable[String], enforcePermission: Boolean = true)
                                  (implicit permissionEnforcer: PermissionEnforcer, sqlDialect: Dialect): ReleasesSqlBuilder = {
    val sqlBuilder: ReleasesSqlBuilder = new ReleasesSqlBuilder()
      .withTitleLike(filters.getTitle)
      .withDates(filters.getQueryStartDate, filters.getQueryEndDate)
      .withKind(filters.getKind())

    if (enforcePermission && !permissionEnforcer.isCurrentUserAdmin && !permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
      checkPermissionSupportedinScopedToken(VIEW_RELEASE, VIEW_WORKFLOW_EXECUTION)
      if (filters.getKind() == ReleaseKind.RELEASE) {
        sqlBuilder.withPermission(VIEW_RELEASE, principals, roleIds)
      } else {
        sqlBuilder.withPermissionOrOwner(VIEW_WORKFLOW_EXECUTION, principals, roleIds)
      }
    }
    if (filters.getStatuses != null && !filters.getStatuses.isEmpty) {
      sqlBuilder.withOneOfStatuses(filters.getStatuses.asScala.toSeq)
    } else {
      sqlBuilder.withoutTemplates()
    }
    if (filters.getTags != null && !filters.getTags.isEmpty) {
      sqlBuilder.withTags(filters.getTags.asScala.toSet)
    }
    if (filters.getTaskTags != null && !filters.getTaskTags.isEmpty) {
      sqlBuilder.withAnyOfTaskTags(filters.getTaskTags.asScala.toSet)
    }
    if (filters.getParentId != null) {
      sqlBuilder.withFolder(filters.getParentId)
    }
    if (filters.withOnlyMine()) {
      sqlBuilder.withOwner(Permissions.getAuthenticatedUserName)
    }
    if (filters.withOnlyFlagged()) {
      sqlBuilder.withOneOfFlagStatuses(Seq(ATTENTION_NEEDED, AT_RISK))
    }

    if (filters.getOrderBy != null) {
      val orderDirection = Option(filters.getOrderDirection).map(direction => direction.isAscending)
      sqlBuilder.orderBy(filters.getOrderBy, orderDirection)
    }

    if (filters.getRiskStatusWithThresholds != null) {
      sqlBuilder.withRiskStatus(filters.getRiskStatusWithThresholds)
    }

    sqlBuilder
  }
}
