package com.xebialabs.xlrelease.security.sql.snapshots.service

import com.xebialabs.deployit.core.rest.api.SecurityResource
import com.xebialabs.deployit.security.{PermissionEditor, Role, RoleService}
import com.xebialabs.xlrelease.configuration.AuditReportSettings
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.sql.SqlFolderRepository
import com.xebialabs.xlrelease.repository.{Ids, SecuredCis}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS
import com.xebialabs.xlrelease.security.sql.snapshots.domain._
import com.xebialabs.xlrelease.security.sql.snapshots.repository.SqlRolesSnapshotRepository
import com.xebialabs.xlrelease.service.{ConfigurationService, UserProfileService}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util.Date
import scala.annotation.tailrec
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
@EventListener
class SqlRolesSnapshotService(val eventBus: XLReleaseEventBus,
                              val sqlPermissionsSnapshotRepository: SqlRolesSnapshotRepository,
                              val permissionChecker: PermissionChecker,
                              val roleService: RoleService,
                              val securityResource: SecurityResource,
                              val permissionEditor: PermissionEditor,
                              val securedCis: SecuredCis,
                              val userProfileService: UserProfileService,
                              val sqlFolderRepository: SqlFolderRepository,
                              val configurationService: ConfigurationService
                             ) extends PermissionsSnapshotService with Logging {

  @AsyncSubscribe
  def onGlobalRolesOrPermissionsUpdated(ev: GlobalRolesOrPermissionsUpdatedEvent): Unit = makeSnapshot()

  @AsyncSubscribe
  def onTeamsUpdated(ev: TeamsUpdatedEvent): Unit = makeSnapshot(Some(ev.containerId))

  override def makeSnapshot(containerId: Option[String] = None): Unit = {
    if (shouldGeneratePermissionSnapshot) {
      generateSnapshot(containerId) match {
        case Some(snapshot) =>
          sqlPermissionsSnapshotRepository.saveSnapshot(snapshot)
          val snapshotCreatedEvent = RolesSnapshotCreated(snapshot.containerIdName, snapshot.parentContainerIdName, snapshot.isInherited, snapshot.snapshotDate)
          eventBus.publish(snapshotCreatedEvent)
        case None =>
      }
    }
  }

  private def generateSnapshot(containerId: Option[String]): Option[RolesSnapshot] = {
    val containerIdName = containerId.map(Ids.getName)
    val parentContainerIdName = containerId.filter(!Ids.isRoot(_)).map(Ids.getParentId).filter(!Ids.isRoot(_)).map(Ids.getName)
    // do not save snapshots when containerId is a release inside a folder
    if (!containerId.exists(id => Ids.isReleaseId(id) && Ids.isInFolder(id))) {
      val containerSecurityUid = getSecurityUID(containerId)
      val isInherited = getInherited(containerId)

      if (isInherited || containerSecurityUid.isEmpty) {
        Some(RolesSnapshot(new Date(), containerIdName, None, isInherited, parentContainerIdName))
      } else {
        val rolePermissionMultimap = permissionEditor.readPermissions(containerSecurityUid.get, false).asScala
        val roles: mutable.Seq[Role] = containerId.map(_ => roleService.readRoleAssignments(containerSecurityUid.get).asScala)
          .getOrElse(roleService.readRoleAssignments().asScala)
        Some(RolesSnapshot(
          new Date(),
          containerIdName,
          Some(roles.map(r => RoleWithPrincipalsAndPermissions(
            r.getId,
            r.getName,
            r.getPrincipals.asScala.toSeq,
            r.getRoles.asScala.toSeq,
            rolePermissionMultimap.get(r).map(_.asScala.map(_.getPermissionName).toSeq).getOrElse(Seq())
          )).toSeq),
          isInherited = false,
          parentContainerIdName
        ))
      }
    } else {
      None
    }

  }

  /**
   *
   * @param containerId - ContainerId that snapshots should be retrieved for for, null for global snapshots
   * @param startDate   - The startDate of the snapshot sequence to retrieve for the container id
   * @param endDate     - THe endDate for the snapshot sequence to retrieve for the container id
   * @return Seq[RolesSnapshot] - Sequence of snapshots that were retrieved for the container id in the given span
   *
   *         This method is responsible for retrieving a collection of snapshot for the given span. The snapshot
   *         sequence will contain the snapshot prior to or equal to the start date up to the end date.
   *
   */
  override def findSnapshotsForRangeWithoutPermissionsCheck(containerId: Option[String], startDate: Date, endDate: Date): Seq[RolesSnapshot] = {
    val startingSnapshotInRange = sqlPermissionsSnapshotRepository.findFirstSnapshot(containerId.map(Ids.getName), startDate);
    startingSnapshotInRange match {
      case Some(firstSnapshot) =>
        sqlPermissionsSnapshotRepository.findSnapshotsForRange(containerId.map(Ids.getName), firstSnapshot.snapshotDate, endDate)
      case _ => Seq.empty[RolesSnapshot]
    }
  }

  override def getMergedContainerSnapshotDataForReport(containerId: String, fromDate: Date, toDate: Date): (Seq[ContainerTeamsAndRolesForPrincipalAndPermission], Seq[RolesSnapshotUnavailableError]) = {
    mapDataForReport(Some(containerId), fromDate, toDate)(buildContainerTeamsAndRolesForPrincipalAndPermission)
  }

  private def buildContainerTeamsAndRolesForPrincipalAndPermission: (String, String, Seq[String]) => ContainerTeamsAndRolesForPrincipalAndPermission = {
    (principal: String, permission: String, roles: Seq[String]) =>
      ContainerTeamsAndRolesForPrincipalAndPermission(
        principal,
        Option(userProfileService.findByUsername(principal)).map(_.getFullName).getOrElse(principal), // this is using cache
        permission,
        roles
      )
  }

  //Find the container defining permissions which is the first container with non-inherited permissions
  @tailrec
  private def findPermissionContainer(containerId: Option[String]): Option[String] = {
    if (getInherited(containerId)) {
      val parentId = containerId.filter(!Ids.isRoot(_)).map(Ids.getParentId)
      findPermissionContainer(parentId)
    } else {
      containerId
    }
  }

  private def getSnapshotSequence(
                                   containerId: Option[String],
                                   fromDate: Date,
                                   toDate: Date,
                                   errors: mutable.ListBuffer[RolesSnapshotUnavailableError]
                                 ): Seq[RolesSnapshot] = {
    val snapshotSequence = findSnapshotsForRangeWithoutPermissionsCheck(containerId, fromDate, toDate)
    logger.trace(s"Retrieved ${snapshotSequence.size} snapshots for containerIs ${containerId} from ${fromDate} to ${toDate}")
    if (snapshotSequence.isEmpty) {
      var errorSnapshot = Option.empty[RolesSnapshot]
      try {
        errorSnapshot = generateSnapshot(findPermissionContainer(containerId))
      } catch {
        // log and kill exception
        case e: Exception => logger.error(
          s"Unable to find any snapshot for folder [${containerId.getOrElse("/")}] " +
            s"and was unable to create new snapshot", e)
      }
      if (shouldGeneratePermissionSnapshot) {
        errors += RolesSnapshotUnavailableError(containerId, fromDate, errorSnapshot.map(_.snapshotDate))
      }
      Seq[RolesSnapshot](errorSnapshot.get)
    } else {
      snapshotSequence
    }
  }

  /**
   *
   * @param containerId - Container id to generate report info for
   * @param fromDate    - Start date for snapshots to report on for the container id
   * @param toDate      - End date for snapshots to report on for the container id
   * @param resultData  - ResultDate to populate with reporting data for the container id
   * @param errors      - Errors encountered during the report data collection
   *
   *                    The method is responsible for collecting report data for the container. The method will collect
   *                    snapshots for the container and global snapshots for the specified time period and map the
   *                    permission data for reporting. If the container is inheriting permissions from a parent folder
   *                    this method will be recursively called for the parent.
   *
   */
  private def mapDataForReportInto(
                                    containerId: Option[String],
                                    fromDate: Date,
                                    toDate: Date,
                                    resultData: mutable.Map[(String, String), mutable.LinkedHashSet[String]],
                                    errors: mutable.ListBuffer[RolesSnapshotUnavailableError]
                                  ): Unit = {
    val snapshotSequence = getSnapshotSequence(containerId, fromDate, toDate, errors)
    val sequenceDates = mapSnapshotDatesInSequence(snapshotSequence)

    //Only fetch global snapshots if we have at least one snapshot for the container that is not inherited
    val globalSnapshotSequence: Seq[RolesSnapshot] = snapshotSequence.filterNot(_.isInherited)
      .headOption
      .map(snapshot => getSnapshotSequence(None, snapshot.snapshotDate, toDate, errors))
      .getOrElse(Seq.empty[RolesSnapshot])

    snapshotSequence.foreach { snapshot =>
      if (snapshot.isInherited) {
        if (snapshot.parentContainerIdName.isDefined) {
          mapDataForReportInto(snapshot.parentContainerIdName, fromDate, sequenceDates.getOrElse(snapshot.snapshotDate, toDate), resultData, errors)
        } else {
          logger.error(s"Strange snapshot found for [${containerId.getOrElse("/")}] which has isInherited but does not have parent, skipping")
        }
      }
      else {
        if (containerId.isDefined) {
          val filteredGlobalSnapshots = globalSnapshotSequence.filter(
            globalSnapshot => globalSnapshot.snapshotDate.before(snapshot.snapshotDate))
          val globalSnapshotStartDate = if (filteredGlobalSnapshots.isEmpty) {
            //Handle error case where no snapshots found and global created after snapshot
            globalSnapshotSequence.head.snapshotDate
          } else {
            filteredGlobalSnapshots.last.snapshotDate
          }

          val filteredGlobalList = sequenceDates.get(snapshot.snapshotDate) match {
            case Some(filterEndDate) =>
              // Filter global snapshots from the current snapshot date to the next snapshot in the sequence
              globalSnapshotSequence.filter(globalSnapshot => (globalSnapshot.snapshotDate.after(globalSnapshotStartDate)
                || globalSnapshot.snapshotDate.equals(globalSnapshotStartDate)) && globalSnapshot.snapshotDate.before(filterEndDate))
            case None =>
              // This is the last snapshot in the sequence, filter global snapshots from the snapshot date
              globalSnapshotSequence.filter(gs => gs.snapshotDate.after(globalSnapshotStartDate)
                || gs.snapshotDate.equals(globalSnapshotStartDate))
          }

          filteredGlobalList.foreach(globalSnapshot => {
            populateResultData(resultData, containerId, snapshot, Some(globalSnapshot))
          })
        } else {
          // This is a global snapshot
          populateResultData(resultData, containerId, snapshot)
        }
      }
    }
  }

  private def mapDataForReport[A](containerId: Option[String], fromDate: Date, toDate: Date)
                                 (mapF: (String, String, Seq[String]) => A): (Seq[A], Seq[RolesSnapshotUnavailableError]) = {
    val resultData = mutable.Map[(String, String), mutable.LinkedHashSet[String]]()
    val errors = mutable.ListBuffer[RolesSnapshotUnavailableError]()
    mapDataForReportInto(containerId, fromDate, toDate, resultData, errors)
    (
      resultData.toSeq.map {
        case ((principal, permission), roles) => mapF(principal, permission, roles.toSeq)
      },
      errors.toSeq
    )
  }

  private type ResultDataMap = mutable.Map[(String, String), mutable.LinkedHashSet[String]]

  private def populateResultData(resultData: ResultDataMap,
                                 containerId: Option[String],
                                 snapshot: RolesSnapshot,
                                 globalSnapshot: Option[RolesSnapshot] = None): Unit = {
    logger.trace(s"Updating result date for container${containerId} with snapshot dated ${snapshot.snapshotDate} " +
      s"and global global snapshot dated ${globalSnapshot.map(_.snapshotDate.getTime)}")
    val globalRoleNameToPrincipals = globalSnapshot
      .flatMap(_.rolesAndPermissions)
      .foldLeft(Map.empty[String, Set[String]]) {
        case (acc, rolesSnapshot) =>
          var result = acc
          rolesSnapshot.foreach { roleWithPrincipalsAndPermissions =>
            val roleName = roleWithPrincipalsAndPermissions.name
            result += roleName -> (result.getOrElse(roleName, Set.empty) ++ roleWithPrincipalsAndPermissions.principals.toSet)
          }
          result
      }

    snapshot.rolesAndPermissions.foreach(_.foreach(role => {
      role.permissions.foreach(permission => {
        val principals = role.principals.toSet
        principals.foreach(principal => {
          resultData.getOrElseUpdate((principal, permission), mutable.LinkedHashSet()).add(role.name)
        })
        if (containerId.isDefined) {
          val globalPrincipals = role.roleIds.flatMap(globalRoleNameToPrincipals.getOrElse(_, Set.empty)).toSet
          role.roleIds.map(roleName => {
            globalRoleNameToPrincipals.getOrElse(roleName, Set.empty).foreach(principal => {
              resultData.getOrElseUpdate((principal, permission), mutable.LinkedHashSet()).add(s"${role.name} ($roleName)")
            })
          })
        }
      })
    }))
  }

  private def mapSnapshotDatesInSequence(seq: Seq[RolesSnapshot]): Map[Date, Date] = {
    // Map the dates in the snapshot sequence so the snapshot date boundaries can be retrieved
    seq.sliding(2).collect {
      case Seq(first, second) => first.snapshotDate -> second.snapshotDate
    }.toMap
  }

  private def shouldGeneratePermissionSnapshot = {
    Try(configurationService.read(AuditReportSettings.AUDIT_REPORT_SETTINGS_ID).asInstanceOf[AuditReportSettings]).toOption.exists(_.isGeneratePermissionSnapshots)
  }

  private def getInherited(containerId: Option[String]): Boolean = {
    containerId.exists { id =>
      Try(sqlFolderRepository.isFolderInherited(id)).getOrElse(false)
    }
  }

  private def getSecurityUID(containerId: Option[String]): Option[String] =
    containerId.flatMap(id =>
      Try(securedCis.getEffectiveSecuredCi(id).getSecurityUid).toOption
    ).orElse(Some(GLOBAL_SECURITY_ALIAS))
}
