package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.{Comment, Release, Task}
import com.xebialabs.xlrelease.repository.CommentRepository
import com.xebialabs.xlrelease.repository.Ids.getParentId
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.CommentRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, CommentPersistence, TaskPersistence}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.joda.time.DateTime

@IsTransactional
class SqlCommentRepository(val taskPersistence: TaskPersistence,
                           val commentPersistence: CommentPersistence,
                           val repositoryAdapter: SqlRepositoryAdapter)
  extends CommentRepository with Logging with DeserializationSupport {

  @Timed
  override def exists(commentId: CiId): Boolean = {
    logger.debug(s"exists($commentId)")
    val taskId = getParentId(commentId)
    withTask0(taskId)(commentPersistence.exists(_, commentId)).getOrElse(false)
  }

  @Timed
  override def create(created: Comment): Boolean = {
    logger.debug(s"create(${created.getId})")
    val taskId = getParentId(created.getId)
    withTask(taskId)(commentPersistence.create(_, CommentRow.fromComment(created)))
  }

  @Timed
  override def createAll(taskId: CiId, comments: Seq[Comment]): Int = {
    val aliens = comments.filter(!_.getId.startsWith(taskId))
    if (aliens.nonEmpty) {
      throw new IllegalArgumentException(s"${aliens.length} comment(s) don't belong to task [$taskId]")
    } else {
      withTask(taskId)(commentPersistence.createAll(_, comments.map(CommentRow.fromComment)))
    }
  }

  @Timed
  override def update(commentId: CiId, updated: String, modified: Option[DateTime]): Boolean = {
    logger.debug(s"update($commentId, $updated)")
    val taskId = getParentId(commentId)
    withTask(taskId)(commentPersistence.update(_, commentId, updated, modified))
  }

  @Timed
  override def findById(commentId: CiId): Comment = {
    logger.debug(s"findById($commentId)")
    val taskId = getParentId(commentId)
    withTask(taskId) { taskUid =>
      commentPersistence.read(taskUid, commentId).map(_.toComment(taskId)).getOrElse {
        logger.warn(s"Comment [$commentId] not found in database")
        throw new NotFoundException(s"Comment [$commentId] not found.")
      }
    }
  }

  @Timed
  override def findByTask(taskId: String): Seq[Comment] = {
    logger.debug(s"findByTask($taskId)")
    withTask(taskId) { taskUid =>
      val found = commentPersistence.readAll(taskUid).map(_.toComment(taskId))
      logger.debug(s"Found ${found.size} comments for $taskId")
      found
    }
  }

  @Timed
  override def deleteByTask(task: Task): Unit = {
    Option(task.getCiUid) match {
      case Some(taskUid) => commentPersistence.deleteAll(taskUid)
      case None => {
        withTask(task.getId) { taskUid =>
          commentPersistence.deleteAll(taskUid)
        }
      }
    }
  }

  @Timed
  override def delete(comment: Comment): Unit = {
    logger.debug(s"delete(${comment.getId})")
    val taskId = getParentId(comment.getId)
    withTask(taskId) { taskUid =>
      commentPersistence.delete(taskUid, comment.getId)
    }
  }

  @Timed
  override def decorate(release: Release): Release = {
    logger.debug(s"decorate(${release.getId})")
    commentPersistence.decorate(release)
  }

  @Timed
  override def decorate(task: Task): Task = {
    logger.debug(s"decorate(${task.getId})")
    Option(task.getCiUid) match {
      case Some(taskUid) => commentPersistence.decorate(taskUid, task)
      case None => {
        withTask(task.getId) { taskUid =>
          commentPersistence.decorate(taskUid, task)
        }
      }
    }

  }

  private def withTask[A](taskId: CiId)(f: CiUid => A): A =
    withTask0(taskId)(f).getOrElse {
      // REL-6288
      val task = repositoryAdapter.read[Task](taskId)
      logger.warn(s"Task with id [$taskId] not found in the database, reinserting...")
      f(taskPersistence.insert(task, task.getRelease.getCiUid))
    }

  private def withTask0[A](taskId: CiId)(f: CiUid => A): Option[A] =
    taskPersistence.getTaskUidById(taskId).map(f)

}
