package com.xebialabs.xlrelease.spring.config

import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.license.service.{DefaultLicenseCiCounterFactory, LicenseService, LicenseServiceFactory}
import com.xebialabs.license.{License, LicenseValidationFilter}
import com.xebialabs.xlplatform.repository.sql.SqlRepositoryMetadataService
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService
import com.xebialabs.xlrelease.config.{ArchivingSettingsManager, XlrConfig}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.features.PersonalAccessTokenFeature
import com.xebialabs.xlrelease.license.XLRLicenseInstallService
import com.xebialabs.xlrelease.repository.sql.persistence.ReleaseCacheEventHandler
import com.xebialabs.xlrelease.repository.{LicenseRepository, ReleaseVariableRepository}
import com.xebialabs.xlrelease.security.{PermissionChecker, PermissionCheckerHolder}
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.views.converters.TasksViewConverter
import io.micrometer.core.instrument.MeterRegistry
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration, Lazy}

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Configuration
class SqlServiceConfiguration(xlrConfig: XlrConfig,
                              sqlRepositories: SqlRepositoriesConfiguration,
                              sqlConfiguration: SqlConfiguration,
                              sqlSecurityConfiguration: SqlSecurityConfiguration,
                              sqlFoldersConfiguration: SqlFoldersConfiguration,
                              sqlPersistenceConfiguration: SqlPersistenceConfiguration, // should not use persistence
                              eventBus: XLReleaseEventBus,
                              meterRegistry: MeterRegistry,
                              releaseVariableRepository: ReleaseVariableRepository) {

  @Autowired var decoratorService: InternalMetadataDecoratorService = _
  @Autowired var permissionEnforcer: PermissionEnforcer = _
  @Autowired var teamService: TeamService = _
  @Autowired var archivingService: ArchivingService = _
  @Autowired var archivingManager: ArchivingSettingsManager = _
  @Autowired
  @Lazy var taskService: TaskService = _
  @Autowired var permissionChecker: PermissionChecker = _
  @Autowired var broadcastService: BroadcastService = _
  @Autowired var licenseRepository: LicenseRepository = _

  @Bean
  def permissionCheckerHolder(): PermissionCheckerHolder.type = {
    PermissionCheckerHolder.init(permissionChecker)
    PermissionCheckerHolder
  }

  @Bean
  def preArchiveService(): PreArchiveService = {
    new PreArchiveService(xlrConfig, archivingService, sqlRepositories.releaseRepository(), archivingManager, meterRegistry)
  }

  @Bean
  def releaseSearchService(): ReleaseSearchService = {
    new SqlReleaseSearchService(
      sqlPersistenceConfiguration.releasePersistence(),
      archivingService,
      sqlPersistenceConfiguration.sqlRepositoryAdapter(),
      teamService,
      decoratorService,
      sqlConfiguration.xlrRepositorySqlDialect(),
      permissionEnforcer,
      sqlSecurityConfiguration.roleService(),
      sqlRepositories.securedCis(),
      sqlRepositories.releaseExtensionsRepository())
  }

  @Bean
  def taskSearchService(taskAccessService: TaskAccessService,
                        tasksViewConverter: TasksViewConverter): TaskSearchService = {
    new SqlTaskSearchService(
      sqlSecurityConfiguration.roleService(),
      taskAccessService,
      decoratorService,
      sqlPersistenceConfiguration.taskPersistence(),
      permissionEnforcer
    )
  }

  @Bean
  //noinspection ScalaStyle
  def folderService(folderOperationValidators: JList[_ <: FolderOperationValidator]): FolderService = {
    new FolderService(sqlFoldersConfiguration.folders(),
      teamService,
      sqlRepositories.releaseRepository(),
      taskService,
      sqlRepositories.cachingConfigurationRepository(),
      releaseVariableRepository,
      sqlConfiguration.ciIdService(),
      releaseSearchService(),
      permissionChecker,
      eventBus,
      archivingService,
      folderOperationValidators.asScala.toSeq,
      broadcastService)
  }

  @Bean
  def repositoryMetadataService() = new SqlRepositoryMetadataService(sqlConfiguration.xlrRepositoryDatabase())

  @Bean
  def licenseService(): LicenseService = {
    new LicenseServiceFactory(repositoryMetadataService(),
      new DefaultLicenseCiCounterFactory(),
      xlrConfig.rootConfig,
      xlrConfig.license_location,
      License.PRODUCT_XL_RELEASE
    ).createLicenseService()
  }

  @Bean
  def licenseCheckFilter(): LicenseValidationFilter = {
    val licenseValidationFilter = new LicenseValidationFilter
    licenseValidationFilter.setLicenseService(licenseService())
    licenseValidationFilter
  }

  @Bean
  def licenseInstallService(): XLRLicenseInstallService = {
    new XLRLicenseInstallService(broadcastService, licenseService(), licenseRepository)
  }

  @Bean
  def userTokenService(personalAccessTokenFeature: PersonalAccessTokenFeature): UserTokenService = {
    new UserTokenService(eventBus, sqlRepositories.userTokenRepository(), personalAccessTokenFeature)
  }

  @Bean
  def releaseCacheEventHandler(): ReleaseCacheEventHandler = {
    new ReleaseCacheEventHandler(sqlConfiguration.releaseCacheService())
  }

  @Bean
  def categoryService(): CategoryService = {
    new CategoryService(sqlRepositories.categoryRepository(), eventBus)
  }

  @Bean
  def workflowService(): WorkflowService = {
    new WorkflowService(sqlRepositories.workflowRepository())
  }
}
