package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.repository.UserProfileRepository
import com.xebialabs.xlrelease.repository.sql.persistence.UserProfilePersistence
import com.xebialabs.xlrelease.repository.sql.query.UserProfileQueryBuilder
import com.xebialabs.xlrelease.views.users.UserFilters
import org.springframework.data.domain.{Pageable, Page => SPage}
import org.springframework.jdbc.core.JdbcTemplate

import java.util.{Date, List => JList}
import scala.jdk.CollectionConverters._
import scala.util.Try

@IsTransactional
class SqlUserProfileRepository(userProfilePersistence: UserProfilePersistence,
                               dialect: Dialect,
                               jdbcTemplate: JdbcTemplate
                              ) extends UserProfileRepository {

  override def create(userProfile: UserProfile): Unit = userProfilePersistence.insert(userProfile)

  override def update(userProfiles: JList[UserProfile]): List[Try[UserProfile]] = {
    userProfiles.asScala.map(profile => Try(userProfilePersistence.updateUserProfile(profile))).toList
  }

  override def updateLastActive(canonicalId: String, lastActive: Date, evictCache: Boolean): Boolean = {
    userProfilePersistence.updateLastActive(canonicalId, lastActive)
  }

  override def updateLastActiveBatch(entries: Map[String, Date]): Int = {
    userProfilePersistence.updateLastActiveBatch(entries)
  }

  override def delete(userProfileId: String): Unit = userProfilePersistence.delete(userProfileId)

  override def findByIdBypassCache(canonicalId: String): Option[UserProfile] = userProfilePersistence.findById(canonicalId)

  override def findById(canonicalId: String): Option[UserProfile] = userProfilePersistence.findById(canonicalId)

  override def findAll(fullProfile: Boolean = false): JList[UserProfile] = userProfilePersistence.findAll(fullProfile).asJava

  @IsReadOnly
  override def exists(canonicalId: String): Boolean = userProfilePersistence.exists(canonicalId)

  override def customSearch(email: String, fullName: String, loginAllowed: java.lang.Boolean,
                            lastActiveAfter: Date, lastActiveBefore: Date, pageOption: Option[Long], resultsPerPageOption: Option[Long]): JList[UserProfile] = {
    userProfilePersistence.search(
      email,
      fullName,
      loginAllowed,
      lastActiveAfter,
      lastActiveBefore,
      resultsPerPageOption,
      pageOption.flatMap(page => resultsPerPageOption.map(resultsPerPage => page * resultsPerPage))
    ).asJava
  }

  @IsReadOnly
  override def countUserWithLoginAllowed(): Int = userProfilePersistence.countUserWithLoginAllowed()

  def searchUserProfiles(userFilters: UserFilters, pageable: Pageable): SPage[UserProfile] = {
    UserProfileQueryBuilder(dialect, jdbcTemplate).from(userFilters).withPageable(pageable).build().execute()
  }
}
